﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/cleanroomsml/CleanRoomsML_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace CleanRoomsML {
namespace Model {

/**
 * <p>Defines an incremental training data channel that references a previously
 * trained model. Incremental training allows you to update an existing trained
 * model with new data, building upon the knowledge from a base model rather than
 * training from scratch. This can significantly reduce training time and
 * computational costs while improving model performance with additional
 * data.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/cleanroomsml-2023-09-06/IncrementalTrainingDataChannel">AWS
 * API Reference</a></p>
 */
class IncrementalTrainingDataChannel {
 public:
  AWS_CLEANROOMSML_API IncrementalTrainingDataChannel() = default;
  AWS_CLEANROOMSML_API IncrementalTrainingDataChannel(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLEANROOMSML_API IncrementalTrainingDataChannel& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_CLEANROOMSML_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the base trained model to use for
   * incremental training. This model serves as the starting point for the
   * incremental training process.</p>
   */
  inline const Aws::String& GetTrainedModelArn() const { return m_trainedModelArn; }
  inline bool TrainedModelArnHasBeenSet() const { return m_trainedModelArnHasBeenSet; }
  template <typename TrainedModelArnT = Aws::String>
  void SetTrainedModelArn(TrainedModelArnT&& value) {
    m_trainedModelArnHasBeenSet = true;
    m_trainedModelArn = std::forward<TrainedModelArnT>(value);
  }
  template <typename TrainedModelArnT = Aws::String>
  IncrementalTrainingDataChannel& WithTrainedModelArn(TrainedModelArnT&& value) {
    SetTrainedModelArn(std::forward<TrainedModelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The version identifier of the base trained model to use for incremental
   * training. If not specified, the latest version of the trained model is used.</p>
   */
  inline const Aws::String& GetVersionIdentifier() const { return m_versionIdentifier; }
  inline bool VersionIdentifierHasBeenSet() const { return m_versionIdentifierHasBeenSet; }
  template <typename VersionIdentifierT = Aws::String>
  void SetVersionIdentifier(VersionIdentifierT&& value) {
    m_versionIdentifierHasBeenSet = true;
    m_versionIdentifier = std::forward<VersionIdentifierT>(value);
  }
  template <typename VersionIdentifierT = Aws::String>
  IncrementalTrainingDataChannel& WithVersionIdentifier(VersionIdentifierT&& value) {
    SetVersionIdentifier(std::forward<VersionIdentifierT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the incremental training data channel. This name is used to
   * identify the channel during the training process and must be unique within the
   * training job.</p>
   */
  inline const Aws::String& GetChannelName() const { return m_channelName; }
  inline bool ChannelNameHasBeenSet() const { return m_channelNameHasBeenSet; }
  template <typename ChannelNameT = Aws::String>
  void SetChannelName(ChannelNameT&& value) {
    m_channelNameHasBeenSet = true;
    m_channelName = std::forward<ChannelNameT>(value);
  }
  template <typename ChannelNameT = Aws::String>
  IncrementalTrainingDataChannel& WithChannelName(ChannelNameT&& value) {
    SetChannelName(std::forward<ChannelNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_trainedModelArn;
  bool m_trainedModelArnHasBeenSet = false;

  Aws::String m_versionIdentifier;
  bool m_versionIdentifierHasBeenSet = false;

  Aws::String m_channelName;
  bool m_channelNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace CleanRoomsML
}  // namespace Aws
