// Copyright (c) Microsoft Corporation.  All Rights Reserved.  See License.txt in the project root for license information.

// This namespace contains constructs for reflecting on the representation of
// F# values and types. It augments the design of System.Reflection.
namespace Microsoft.FSharp.Reflection

open System
open System.Reflection
open System.Diagnostics.CodeAnalysis
open Microsoft.FSharp.Core
open Microsoft.FSharp.Primitives.Basics
open Microsoft.FSharp.Collections

/// <summary>Represents a case of a discriminated union type</summary>
///
/// <namespacedoc><summary>
///   Library functionality for accessing additional information about F# types and F# values at
///   runtime, augmenting that available through <a href="https://learn.microsoft.com/dotnet/api/system.reflection">System.Reflection</a>.
/// </summary></namespacedoc>
[<Sealed>]
type UnionCaseInfo =
    /// <summary>The name of the case.</summary>
    ///
    /// <example id="Name-1">
    /// <code lang="fsharp">
    /// type Weather = Rainy | Sunny
    ///
    /// typeof&lt;Weather&gt;
    /// |> FSharpType.GetUnionCases
    /// |> Array.map (fun x -> x.Name)
    /// </code>
    /// Evaluates to <c>[|"Rainy", "Sunny"|]</c>
    /// </example>
    member Name: string

    /// <summary>The type in which the case occurs.</summary>
    ///
    /// <example id="DeclaringType-1">
    /// <code lang="fsharp">
    /// type Weather = Rainy | Sunny
    ///
    /// let rainy =
    ///     typeof&lt;Weather&gt;
    ///     |> FSharpType.GetUnionCases
    ///     |> Array.head
    ///
    /// rainy.DeclaringType
    /// </code>
    /// Evaluates to a value of type <c>System.Type</c>
    /// that holds type information for <c>Weather</c>.
    /// </example>
    member DeclaringType: Type

    /// <summary>Returns the custom attributes associated with the case.</summary>
    /// <returns>An array of custom attributes.</returns>
    ///
    /// <example id="GetCustomAttributes-1">
    /// <code lang="fsharp">
    /// type Weather =
    ///     | Rainy
    ///     | Sunny
    ///
    /// typeof&lt;Weather&gt;
    /// |> FSharpType.GetUnionCases
    /// |> Array.map (fun x -> x.GetCustomAttributes())
    /// </code>
    /// Evaluates to
    /// <code lang="fsharp">
    /// [|[|Microsoft.FSharp.Core.CompilationMappingAttribute
    ///     {ResourceName = null;
    ///      SequenceNumber = 0;
    ///      SourceConstructFlags = UnionCase;
    ///      TypeDefinitions = null;
    ///      TypeId = Microsoft.FSharp.Core.CompilationMappingAttribute;
    ///      VariantNumber = 0;}|];
    /// [|Microsoft.FSharp.Core.CompilationMappingAttribute
    ///     {ResourceName = null;
    ///      SequenceNumber = 1;
    ///      SourceConstructFlags = UnionCase;
    ///      TypeDefinitions = null;
    ///      TypeId = Microsoft.FSharp.Core.CompilationMappingAttribute;
    ///      VariantNumber = 0;}|]|]
    /// </code>
    /// </example>
    member GetCustomAttributes: unit -> obj array

    /// <summary>Returns the custom attributes associated with the case matching the given attribute type.</summary>
    /// <param name="attributeType">The type of attributes to return.</param>
    ///
    /// <returns>An array of custom attributes.</returns>
    ///
    /// <example id="GetCustomAttributes-2">
    /// <code lang="fsharp">
    /// type Signal(signal: string) =
    ///    inherit System.Attribute()
    ///    member this.Signal = signal
    ///
    /// type Answer =
    ///     | [&lt;Signal("Thumbs up")&gt;] Yes
    ///     | [&lt;Signal("Thumbs down")&gt;] No
    ///
    /// typeof&lt;Answer&gt;
    /// |> FSharpType.GetUnionCases
    /// |> Array.map (fun x -> x.GetCustomAttributes(typeof&lt;Signal&gt;))
    /// </code>
    /// Evaluates to
    /// <code lang="fsharp">
    /// [|[|FSI_0147+Signal {Signal = "Thumbs up";
    ///                      TypeId = FSI_0147+Signal;}|];
    ///   [|FSI_0147+Signal {Signal = "Thumbs down";
    ///                      TypeId = FSI_0147+Signal;}|]|]
    /// </code>
    /// </example>
    member GetCustomAttributes: attributeType: Type -> obj array

    /// <summary>Returns the custom attributes data associated with the case.</summary>
    /// <returns>An list of custom attribute data items.</returns>
    ///
    /// <example id="GetCustomAttributesData-1">
    /// <code lang="fsharp">
    /// type Signal(signal: string) =
    ///   inherit System.Attribute()
    ///   member this.Signal = signal
    ///
    /// type Answer =
    ///     | [&lt;Signal("Thumbs up")&gt;] Yes
    ///     | [&lt;Signal("Thumbs down")&gt;] No
    ///
    /// let answerYes =
    ///     typeof&lt;Answer&gt;
    ///     |> FSharpType.GetUnionCases
    ///     |> Array.find (fun x -> x.Name = "Yes")
    ///
    /// answerYes.GetCustomAttributesData()
    /// </code>
    /// Evaluates to
    /// <code lang="fsharp">
    ///  [|[FSI_0150+Signal("Thumbs up")]
    ///      {AttributeType = FSI_0150+Signal;
    ///       Constructor = Void .ctor(System.String);
    ///       ConstructorArguments = seq ["Thumbs up"];
    ///       NamedArguments = seq [];};
    ///    [Microsoft.FSharp.Core.CompilationMappingAttribute((Microsoft.FSharp.Core.SourceConstructFlags)8, (Int32)0)]
    ///      {AttributeType = Microsoft.FSharp.Core.CompilationMappingAttribute;
    ///       Constructor = Void .ctor(Microsoft.FSharp.Core.SourceConstructFlags, Int32);
    ///       ConstructorArguments = seq
    ///                                [(Microsoft.FSharp.Core.SourceConstructFlags)8;
    ///                                 (Int32)0];
    ///       NamedArguments = seq [];}|]
    /// </code>
    /// </example>
    member GetCustomAttributesData: unit -> System.Collections.Generic.IList<CustomAttributeData>

    /// <summary>The fields associated with the case, represented by a PropertyInfo.</summary>
    /// <returns>The fields associated with the case.</returns>
    ///
    /// <example id="GetFields-1">
    /// <code lang="fsharp">
    /// type Shape =
    ///     | Rectangle of width : float * length : float
    ///     | Circle of radius : float
    ///     | Prism of width : float * float * height : float
    ///
    /// typeof&lt;Shape&gt;
    /// |> FSharpType.GetUnionCases
    /// |> Array.map (fun unionCase ->
    ///     unionCase.GetFields()
    ///     |> Array.map (fun fieldInfo ->
    ///         fieldInfo.Name,
    ///         fieldInfo.PropertyType.Name))
    /// </code>
    /// Evaluates to
    /// <code lang="fsharp">
    /// [|[|("width", "Double"); ("length", "Double")|];
    ///   [|("radius", "Double")|];
    ///   [|("width", "Double"); ("Item2", "Double"); ("height", "Double")|]|]
    /// </code>
    /// </example>
    member GetFields: unit -> PropertyInfo array

    /// <summary>The integer tag for the case.</summary>
    ///
    /// <example id="Tag-1">
    /// <code lang="fsharp">
    /// type CoinToss = Heads | Tails
    ///
    /// typeof&lt;CoinToss&gt;
    /// |> FSharpType.GetUnionCases
    /// |> Array.map (fun x -> $"{x.Name} has tag {x.Tag}")
    /// </code>
    /// Evaluates to <c>[|"Heads has tag 0"; "Tails has tag 1"|]</c>
    /// </example>
    member Tag: int

/// <summary>Contains operations associated with constructing and analyzing values associated with F# types
/// such as records, unions and tuples.</summary>
[<AbstractClass; Sealed>]
type FSharpValue =

    /// <summary>Reads a field from a record value.</summary>
    ///
    /// <remarks>Assumes the given input is a record value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    ///
    /// <param name="record">The record object.</param>
    /// <param name="info">The PropertyInfo describing the field to read.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input is not a record value.</exception>
    ///
    /// <returns>The field from the record.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetRecordField: record: obj * info: PropertyInfo -> objnull

    /// <summary>Precompute a function for reading a particular field from a record.
    /// Assumes the given type is a RecordType with a field of the given name.
    /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</summary>
    ///
    /// <remarks>Using the computed function will typically be faster than executing a corresponding call to Value.GetInfo
    /// because the path executed by the computed function is optimized given the knowledge that it will be
    /// used to read values of the given type.</remarks>
    ///
    /// <param name="info">The PropertyInfo of the field to read.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
    ///
    /// <returns>A function to read the specified field from the record.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeRecordFieldReader: info: PropertyInfo -> (obj -> objnull)

    /// <summary>Creates an instance of a record type.</summary>
    ///
    /// <remarks>Assumes the given input is a record type.</remarks>
    ///
    /// <param name="recordType">The type of record to make.</param>
    /// <param name="values">The array of values to initialize the record.</param>
    /// <param name="bindingFlags">Optional binding flags for the record.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
    ///
    /// <returns>The created record.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeRecord:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
        values: objnull array *
        ?bindingFlags: BindingFlags ->
            obj

    /// <summary>Reads all the fields from a record value.</summary>
    ///
    /// <remarks>Assumes the given input is a record value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    /// <param name="record">The record object.</param>
    /// <param name="bindingFlags">Optional binding flags for the record.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
    ///
    /// <returns>The array of fields from the record.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetRecordFields: record: obj * ?bindingFlags: BindingFlags -> objnull array

    /// <summary>Precompute a function for reading all the fields from a record. The fields are returned in the
    /// same order as the fields reported by a call to Microsoft.FSharp.Reflection.Type.GetInfo for
    /// this type.</summary>
    ///
    /// <remarks>Assumes the given type is a RecordType.
    /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.
    ///
    /// Using the computed function will typically be faster than executing a corresponding call to Value.GetInfo
    /// because the path executed by the computed function is optimized given the knowledge that it will be
    /// used to read values of the given type.</remarks>
    ///
    /// <param name="recordType">The type of record to read.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
    ///
    /// <returns>An optimized reader for the given record type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeRecordReader:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
        ?bindingFlags: BindingFlags ->
            (obj -> objnull array)

    /// <summary>Precompute a function for constructing a record value. </summary>
    ///
    /// <remarks>Assumes the given type is a RecordType.
    /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</remarks>
    ///
    /// <param name="recordType">The type of record to construct.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
    ///
    /// <returns>A function to construct records of the given type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeRecordConstructor:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
        ?bindingFlags: BindingFlags ->
            (objnull array -> obj)

    /// <summary>Get a ConstructorInfo for a record type</summary>
    ///
    /// <param name="recordType">The record type.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>A ConstructorInfo for the given record type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeRecordConstructorInfo:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
        ?bindingFlags: BindingFlags ->
            ConstructorInfo

    /// <summary>Create a union case value.</summary>
    ///
    /// <param name="unionCase">The description of the union case to create.</param>
    /// <param name="args">The array of arguments to construct the given case.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>The constructed union case.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeUnion: unionCase: UnionCaseInfo * args: objnull array * ?bindingFlags: BindingFlags -> objnull

    /// <summary>Identify the union case and its fields for an object</summary>
    ///
    /// <remarks>Assumes the given input is a union case value. If not, <see cref="T:System.ArgumentException" /> is raised.
    ///
    /// If the type is not given, then the runtime type of the input object is used to identify the
    /// relevant union type. The type should always be given if the input object may be null. For example,
    /// option values may be represented using the 'null'.</remarks>
    /// <param name="value">The input union case.</param>
    /// <param name="unionType">The union type containing the value.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a union case value.</exception>
    ///
    /// <returns>The description of the union case and its fields.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetUnionFields:
        value: objnull *
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type *
        ?bindingFlags: BindingFlags ->
            UnionCaseInfo * objnull array

    /// <summary>Assumes the given type is a union type.
    /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</summary>
    ///
    /// <remarks>Using the computed function is more efficient than calling GetUnionCase
    /// because the path executed by the computed function is optimized given the knowledge that it will be
    /// used to read values of the given type.</remarks>
    ///
    /// <param name="unionType">The type of union to optimize reading.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>An optimized function to read the tags of the given union type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeUnionTagReader:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type * ?bindingFlags: BindingFlags ->
            (objnull -> int)

    /// <summary>Precompute a property or static method for reading an integer representing the case tag of a union type.</summary>
    ///
    /// <param name="unionType">The type of union to read.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>The description of the union case reader.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeUnionTagMemberInfo:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type * ?bindingFlags: BindingFlags ->
            MemberInfo

    /// <summary>Precompute a function for reading all the fields for a particular discriminator case of a union type</summary>
    ///
    /// <remarks>Using the computed function will typically be faster than executing a corresponding call to GetFields</remarks>
    ///
    /// <param name="unionCase">The description of the union case to read.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>A function to for reading the fields of the given union case.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeUnionReader:
        unionCase: UnionCaseInfo * ?bindingFlags: BindingFlags -> (objnull -> objnull array)

    /// <summary>Precompute a function for constructing a discriminated union value for a particular union case. </summary>
    ///
    /// <param name="unionCase">The description of the union case.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>A function for constructing values of the given union case.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeUnionConstructor:
        unionCase: UnionCaseInfo * ?bindingFlags: BindingFlags -> (objnull array -> objnull)

    /// <summary>A method that constructs objects of the given case</summary>
    ///
    /// <param name="unionCase">The description of the union case.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>The description of the constructor of the given union case.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeUnionConstructorInfo: unionCase: UnionCaseInfo * ?bindingFlags: BindingFlags -> MethodInfo

    /// <summary>Reads all the fields from a value built using an instance of an F# exception declaration</summary>
    ///
    /// <remarks>Assumes the given input is an F# exception value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    ///
    /// <param name="exn">The exception instance.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not an F# exception.</exception>
    ///
    /// <returns>The fields from the given exception.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetExceptionFields: exn: obj * ?bindingFlags: BindingFlags -> objnull array

    /// <summary>Creates an instance of a tuple type</summary>
    ///
    /// <remarks>Assumes at least one element is given. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    ///
    /// <param name="tupleElements">The array of tuple fields.</param>
    /// <param name="tupleType">The tuple type to create.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown if no elements are given.</exception>
    ///
    /// <returns>An instance of the tuple type with the given elements.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeTuple: tupleElements: objnull array * tupleType: Type -> obj

    /// <summary>Reads a field from a tuple value.</summary>
    ///
    /// <remarks>Assumes the given input is a tuple value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    ///
    /// <param name="tuple">The input tuple.</param>
    /// <param name="index">The index of the field to read.</param>
    ///
    /// <returns>The value of the field.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetTupleField: tuple: obj * index: int -> objnull

    /// <summary>Reads all fields from a tuple.</summary>
    ///
    /// <remarks>Assumes the given input is a tuple value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    ///
    /// <param name="tuple">The input tuple.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input is not a tuple value.</exception>
    ///
    /// <returns>An array of the fields from the given tuple.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetTupleFields: tuple: obj -> objnull array

    /// <summary>Precompute a function for reading the values of a particular tuple type</summary>
    ///
    /// <remarks>Assumes the given type is a TupleType.
    /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</remarks>
    ///
    /// <param name="tupleType">The tuple type to read.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the given type is not a tuple type.</exception>
    ///
    /// <returns>A function to read values of the given tuple type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeTupleReader:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] tupleType: Type -> (obj -> objnull array)

    /// <summary>Gets information that indicates how to read a field of a tuple</summary>
    ///
    /// <param name="tupleType">The input tuple type.</param>
    /// <param name="index">The index of the tuple element to describe.</param>
    ///
    /// <returns>The description of the tuple element and an optional type and index if the tuple is big.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeTuplePropertyInfo:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] tupleType: Type * index: int ->
            PropertyInfo * (Type * int) option

    /// <summary>Precompute a function for reading the values of a particular tuple type</summary>
    ///
    /// <remarks>Assumes the given type is a TupleType.
    /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</remarks>
    ///
    /// <param name="tupleType">The type of tuple to read.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the given type is not a tuple type.</exception>
    ///
    /// <returns>A function to read a particular tuple type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeTupleConstructor:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] tupleType: Type -> (objnull array -> obj)

    /// <summary>Gets a method that constructs objects of the given tuple type.
    /// For small tuples, no additional type will be returned.</summary>
    ///
    /// <remarks>For large tuples, an additional type is returned indicating that
    /// a nested encoding has been used for the tuple type. In this case
    /// the suffix portion of the tuple type has the given type and an
    /// object of this type must be created and passed as the last argument
    /// to the ConstructorInfo. A recursive call to PreComputeTupleConstructorInfo
    /// can be used to determine the constructor for that the suffix type.</remarks>
    ///
    /// <param name="tupleType">The input tuple type.</param>
    ///
    /// <returns>The description of the tuple type constructor and an optional extra type
    /// for large tuples.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member PreComputeTupleConstructorInfo:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] tupleType: Type ->
            ConstructorInfo * Type option

    /// <summary>Builds a typed function from object from a dynamic function implementation</summary>
    ///
    /// <param name="functionType">The function type of the implementation.</param>
    /// <param name="implementation">The untyped lambda of the function implementation.</param>
    ///
    /// <returns>A typed function from the given dynamic implementation.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeFunction:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] functionType: Type *
        implementation: (objnull -> objnull) ->
            obj

/// <summary>Contains operations associated with constructing and analyzing F# types such as records, unions and tuples</summary>
[<AbstractClass; Sealed>]
type FSharpType =

    /// <summary>Reads all the fields from a record value, in declaration order</summary>
    ///
    /// <remarks>Assumes the given input is a record value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    ///
    /// <param name="recordType">The input record type.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>An array of descriptions of the properties of the record type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetRecordFields:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
        ?bindingFlags: BindingFlags ->
            PropertyInfo array

    /// <summary>Gets the cases of a union type.</summary>
    ///
    /// <remarks>Assumes the given type is a union type. If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</remarks>
    ///
    /// <param name="unionType">The input union type.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a union type.</exception>
    ///
    /// <returns>An array of descriptions of the cases of the given union type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetUnionCases:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type * ?bindingFlags: BindingFlags ->
            UnionCaseInfo array

    /// <summary>Return true if the <c>typ</c> is a representation of an F# record type </summary>
    ///
    /// <param name="typ">The type to check.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>True if the type check succeeds.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member IsRecord:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] typ: Type * ?bindingFlags: BindingFlags ->
            bool

    /// <summary>Returns true if the <c>typ</c> is a representation of an F# union type or the runtime type of a value of that type</summary>
    ///
    /// <param name="typ">The type to check.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>True if the type check succeeds.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member IsUnion:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] typ: Type * ?bindingFlags: BindingFlags ->
            bool

    /// <summary>Reads all the fields from an F# exception declaration, in declaration order</summary>
    ///
    /// <remarks>Assumes <c>exceptionType</c> is an exception representation type. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
    ///
    /// <param name="exceptionType">The exception type to read.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <exception cref="T:System.ArgumentException">Thrown if the given type is not an exception.</exception>
    ///
    /// <returns>An array containing the PropertyInfo of each field in the exception.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetExceptionFields:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] exceptionType: Type *
        ?bindingFlags: BindingFlags ->
            PropertyInfo array

    /// <summary>Returns true if the <c>typ</c> is a representation of an F# exception declaration</summary>
    ///
    /// <param name="exceptionType">The type to check.</param>
    /// <param name="bindingFlags">Optional binding flags.</param>
    ///
    /// <returns>True if the type check is an F# exception.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member IsExceptionRepresentation:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] exceptionType: Type *
        ?bindingFlags: BindingFlags ->
            bool

    /// <summary>Returns a <see cref="T:System.Type"/> representing the F# function type with the given domain and range</summary>
    ///
    /// <param name="domain">The input type of the function.</param>
    /// <param name="range">The output type of the function.</param>
    ///
    /// <returns>The function type with the given domain and range.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeFunctionType:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] domain: Type *
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] range: Type ->
            Type

    /// <summary>Returns a <see cref="T:System.Type"/> representing an F# tuple type with the given element types</summary>
    ///
    /// <param name="types">An array of types for the tuple elements.</param>
    ///
    /// <returns>The type representing the tuple containing the input elements.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeTupleType: types: Type array -> Type

    /// <summary>Returns a <see cref="T:System.Type"/> representing an F# tuple type with the given element types</summary>
    ///
    /// <param name="asm">Runtime assembly containing System.Tuple definitions.</param>
    /// <param name="types">An array of types for the tuple elements.</param>
    ///
    /// <returns>The type representing the tuple containing the input elements.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeTupleType: asm: Assembly * types: Type array -> Type

    /// <summary>Returns a <see cref="T:System.Type"/> representing an F# struct tuple type with the given element types</summary>
    ///
    /// <param name="asm">Runtime assembly containing System.ValueTuple definitions.</param>
    /// <param name="types">An array of types for the tuple elements.</param>
    ///
    /// <returns>The type representing the struct tuple containing the input elements.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeStructTupleType: asm: Assembly * types: Type array -> Type

    /// <summary>Returns a <see cref="T:System.Type"/> representing an F# struct tuple type with the given element types</summary>
    ///
    /// <param name="types">An array of types for the tuple elements.</param>
    ///
    /// <returns>The type representing the struct tuple containing the input elements.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member MakeStructTupleType: types: Type array -> Type

    /// <summary>Return true if the <c>typ</c> is a representation of an F# tuple type </summary>
    ///
    /// <param name="typ">The type to check.</param>
    ///
    /// <returns>True if the type check succeeds.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member IsTuple: [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] typ: Type -> bool

    /// <summary>Return true if the <c>typ</c> is a representation of an F# function type or the runtime type of a closure implementing an F# function type</summary>
    ///
    /// <param name="typ">The type to check.</param>
    ///
    /// <returns>True if the type check succeeds.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member IsFunction: [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] typ: Type -> bool

    /// <summary>Return true if the <c>typ</c> is a <see cref="T:System.Type"/> value corresponding to the compiled form of an F# module </summary>
    ///
    /// <param name="typ">The type to check.</param>
    ///
    /// <returns>True if the type check succeeds.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member IsModule: [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] typ: Type -> bool

    /// <summary>Gets the tuple elements from the representation of an F# tuple type.</summary>
    ///
    /// <param name="tupleType">The input tuple type.</param>
    ///
    /// <returns>An array of the types contained in the given tuple type.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetTupleElements:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] tupleType: Type -> Type array

    /// <summary>Gets the domain and range types from an F# function type  or from the runtime type of a closure implementing an F# type</summary>
    ///
    /// <param name="functionType">The input function type.</param>
    ///
    /// <returns>A tuple of the domain and range types of the input function.</returns>
    ///
    /// <example-tbd></example-tbd>
    static member GetFunctionElements:
        [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] functionType: Type -> Type * Type

/// <summary>Defines further accessing additional information about F# types and F# values at runtime.</summary>
[<AutoOpen>]
module FSharpReflectionExtensions =
    type FSharpValue with

        /// <summary>Creates an instance of a record type.</summary>
        ///
        /// <remarks>Assumes the given input is a record type.</remarks>
        ///
        /// <param name="recordType">The type of record to make.</param>
        /// <param name="values">The array of values to initialize the record.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flags that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
        ///
        /// <returns>The created record.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member MakeRecord:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
            values: objnull array *
            ?allowAccessToPrivateRepresentation: bool ->
                obj

        /// <summary>Reads all the fields from a record value.</summary>
        ///
        /// <remarks>Assumes the given input is a record value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
        ///
        /// <param name="record">The record object.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
        ///
        /// <returns>The array of fields from the record.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member GetRecordFields:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] record: obj *
            ?allowAccessToPrivateRepresentation: bool ->
                objnull array

        /// <summary>Precompute a function for reading all the fields from a record. The fields are returned in the
        /// same order as the fields reported by a call to Microsoft.FSharp.Reflection.Type.GetInfo for
        /// this type.</summary>
        ///
        /// <remarks>Assumes the given type is a RecordType.
        /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.
        ///
        /// Using the computed function will typically be faster than executing a corresponding call to Value.GetInfo
        /// because the path executed by the computed function is optimized given the knowledge that it will be
        /// used to read values of the given type.</remarks>
        ///
        /// <param name="recordType">The type of record to read.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
        ///
        /// <returns>An optimized reader for the given record type.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeRecordReader:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                (obj -> objnull array)

        /// <summary>Precompute a function for constructing a record value. </summary>
        ///
        /// <remarks>Assumes the given type is a RecordType.
        /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</remarks>
        ///
        /// <param name="recordType">The type of record to construct.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a record type.</exception>
        ///
        /// <returns>A function to construct records of the given type.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeRecordConstructor:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                (objnull array -> obj)

        /// <summary>Get a ConstructorInfo for a record type</summary>
        ///
        /// <param name="recordType">The record type.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>A ConstructorInfo for the given record type.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeRecordConstructorInfo:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                ConstructorInfo

        /// <summary>Create a union case value.</summary>
        ///
        /// <param name="unionCase">The description of the union case to create.</param>
        /// <param name="args">The array of arguments to construct the given case.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>The constructed union case.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member MakeUnion:
            unionCase: UnionCaseInfo * args: objnull array * ?allowAccessToPrivateRepresentation: bool -> objnull

        /// <summary>Identify the union case and its fields for an object</summary>
        ///
        /// <remarks>Assumes the given input is a union case value. If not, <see cref="T:System.ArgumentException" /> is raised.
        ///
        /// If the type is not given, then the runtime type of the input object is used to identify the
        /// relevant union type. The type should always be given if the input object may be null. For example,
        /// option values may be represented using the 'null'.</remarks>
        ///
        /// <param name="value">The input union case.</param>
        /// <param name="unionType">The union type containing the value.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a union case value.</exception>
        ///
        /// <returns>The description of the union case and its fields.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member GetUnionFields:
            value: objnull *
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                UnionCaseInfo * objnull array

        /// <summary>Assumes the given type is a union type.
        /// If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</summary>
        ///
        /// <remarks>Using the computed function is more efficient than calling GetUnionCase
        /// because the path executed by the computed function is optimized given the knowledge that it will be
        /// used to read values of the given type.</remarks>
        ///
        /// <param name="unionType">The type of union to optimize reading.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>An optimized function to read the tags of the given union type.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeUnionTagReader:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                (objnull -> int)

        /// <summary>Precompute a property or static method for reading an integer representing the case tag of a union type.</summary>
        ///
        /// <param name="unionType">The type of union to read.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>The description of the union case reader.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeUnionTagMemberInfo:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                MemberInfo

        /// <summary>Precompute a function for reading all the fields for a particular discriminator case of a union type</summary>
        ///
        /// <remarks>Using the computed function will typically be faster than executing a corresponding call to GetFields</remarks>
        ///
        /// <param name="unionCase">The description of the union case to read.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>A function to for reading the fields of the given union case.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeUnionReader:
            unionCase: UnionCaseInfo * ?allowAccessToPrivateRepresentation: bool -> (objnull -> objnull array)

        /// <summary>Precompute a function for constructing a discriminated union value for a particular union case. </summary>
        ///
        /// <param name="unionCase">The description of the union case.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>A function for constructing values of the given union case.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeUnionConstructor:
            unionCase: UnionCaseInfo * ?allowAccessToPrivateRepresentation: bool -> (objnull array -> objnull)

        /// <summary>A method that constructs objects of the given case</summary>
        ///
        /// <param name="unionCase">The description of the union case.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>The description of the constructor of the given union case.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member PreComputeUnionConstructorInfo:
            unionCase: UnionCaseInfo * ?allowAccessToPrivateRepresentation: bool -> MethodInfo

        /// <summary>Reads all the fields from a value built using an instance of an F# exception declaration</summary>
        ///
        /// <remarks>Assumes the given input is an F# exception value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
        ///
        /// <param name="exn">The exception instance.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown when the input type is not an F# exception.</exception>
        ///
        /// <returns>The fields from the given exception.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member GetExceptionFields: exn: obj * ?allowAccessToPrivateRepresentation: bool -> objnull array

    type FSharpType with

        /// <summary>Reads all the fields from a record value, in declaration order</summary>
        ///
        /// <remarks>Assumes the given input is a record value. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
        ///
        /// <param name="recordType">The input record type.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>An array of descriptions of the properties of the record type.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member GetRecordFields:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] recordType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                PropertyInfo array

        /// <summary>Gets the cases of a union type.</summary>
        ///
        /// <remarks>Assumes the given type is a union type. If not, <see cref="T:System.ArgumentException" /> is raised during pre-computation.</remarks>
        ///
        /// <param name="unionType">The input union type.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown when the input type is not a union type.</exception>
        ///
        /// <returns>An array of descriptions of the cases of the given union type.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member GetUnionCases:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] unionType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                UnionCaseInfo array

        /// <summary>Return true if the <c>typ</c> is a representation of an F# record type </summary>
        ///
        /// <param name="typ">The type to check.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>True if the type check succeeds.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member IsRecord:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] typ: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                bool

        /// <summary>Returns true if the <c>typ</c> is a representation of an F# union type or the runtime type of a value of that type</summary>
        ///
        /// <param name="typ">The type to check.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>True if the type check succeeds.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member IsUnion:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] typ: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                bool

        /// <summary>Reads all the fields from an F# exception declaration, in declaration order</summary>
        ///
        /// <remarks>Assumes <c>exceptionType</c> is an exception representation type. If not, <see cref="T:System.ArgumentException" /> is raised.</remarks>
        ///
        /// <param name="exceptionType">The exception type to read.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <exception cref="T:System.ArgumentException">Thrown if the given type is not an exception.</exception>
        ///
        /// <returns>An array containing the PropertyInfo of each field in the exception.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member GetExceptionFields:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] exceptionType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                PropertyInfo array

        /// <summary>Returns true if the <c>exceptionType</c> is a representation of an F# exception declaration</summary>
        ///
        /// <param name="exceptionType">The type to check.</param>
        /// <param name="allowAccessToPrivateRepresentation">Optional flag that denotes accessibility of the private representation.</param>
        ///
        /// <returns>True if the type check is an F# exception.</returns>
        ///
        /// <example-tbd></example-tbd>
        static member IsExceptionRepresentation:
            [<DynamicallyAccessedMembers(DynamicallyAccessedMemberTypes.All)>] exceptionType: Type *
            ?allowAccessToPrivateRepresentation: bool ->
                bool

module internal ReflectionUtils =
    type BindingFlags = System.Reflection.BindingFlags
    val toBindingFlags: allowAccessToNonPublicMembers: bool -> BindingFlags
