// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

namespace System.Globalization
{
    /// <remarks>
    /// Calendar support range:
    ///     Calendar           Minimum     Maximum
    ///     ==========         ==========  ==========
    ///     Gregorian          918/02/19   2051/02/10
    ///     KoreanLunisolar    918/01/01   2050/13/29
    /// </remarks>
    public class KoreanLunisolarCalendar : EastAsianLunisolarCalendar
    {
        public const int GregorianEra = 1;

        private const int MinLunisolarYear = 918;
        private const int MaxLunisolarYear = 2050;

        private static readonly DateTime s_minDate = new DateTime(918, 2, 19);
        private static readonly DateTime s_maxDate = new DateTime((new DateTime(2051, 2, 10, 23, 59, 59, 999)).Ticks + 9999);

        public override DateTime MinSupportedDateTime => s_minDate;

        public override DateTime MaxSupportedDateTime => s_maxDate;

        protected override int DaysInYearBeforeMinSupportedYear =>
            // 917  -- From http://emr.cs.iit.edu/home/reingold/calendar-book/Calendrica.html
            //              using ChineseLunisolar
            384;

        // Data for years 1391-2050 matches that available from
        // Korea Astronomy and Space Science Institute (KASI)
        // https://astro.kasi.re.kr:444/life/pageView/5
        private static readonly int[,] s_yinfo =
        {
/*Y           LM  Lmon  Lday    DaysPerMonth               D1   D2   D3   D4   D5   D6   D7   D8   D9   D10  D11  D12  D13  #Days
0918     */ { 00,   02,   19,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
0919     */ { 00,   02,   09,   0b0100010111010000 }, /*   29   30   29   29   29   30   29   30   30   30   29   30        354
0920     */ { 06,   01,   29,   0b1010001011011000 }, /*   30   29   30   29   29   29   30   29   30   30   29   30   30   384
0921     */ { 00,   02,   16,   0b1010001010110000 }, /*   30   29   30   29   29   29   30   29   30   29   30   30        354
0922     */ { 00,   02,   05,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
0923     */ { 04,   01,   25,   0b1011010010101000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29   30   384
0924     */ { 00,   02,   13,   0b0110110100100000 }, /*   29   30   30   29   30   30   29   30   29   29   30   29        354
0925     */ { 12,   02,   01,   0b1010110101100000 }, /*   30   29   30   29   30   30   29   30   29   30   30   29   29   384
0926     */ { 00,   02,   20,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
0927     */ { 00,   02,   10,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
0928     */ { 08,   01,   31,   0b0100010110111000 }, /*   29   30   29   29   29   30   29   30   30   29   30   30   30   384
0929     */ { 00,   02,   18,   0b0100010101110000 }, /*   29   30   29   29   29   30   29   30   29   30   30   30        354
0930     */ { 00,   02,   07,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
0931     */ { 05,   01,   27,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30   29   383
0932     */ { 00,   02,   14,   0b1110100101010000 }, /*   30   30   30   29   30   29   29   30   29   30   29   30        355
0933     */ { 00,   02,   03,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
0934     */ { 01,   01,   23,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
0935     */ { 00,   02,   11,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
0936     */ { 11,   02,   01,   0b0101001101100000 }, /*   29   30   29   30   29   29   30   30   29   30   30   29   29   383
0937     */ { 00,   02,   18,   0b1100101011000000 }, /*   30   30   29   29   30   29   30   29   30   30   29   29        354
0938     */ { 00,   02,   07,   0b1110010101100000 }, /*   30   30   30   29   29   30   29   30   29   30   30   29        355
0939     */ { 07,   01,   28,   0b1101001010101000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29   30   384
0940     */ { 00,   02,   16,   0b1101001010100000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29        354
0941     */ { 00,   02,   04,   0b1101100101010000 }, /*   30   30   29   30   30   29   29   30   29   30   29   30        355
0942     */ { 03,   01,   25,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
0943     */ { 00,   02,   13,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
0944     */ { 12,   02,   02,   0b1010011011010000 }, /*   30   29   30   29   29   30   30   29   30   30   29   30   29   384
0945     */ { 00,   02,   20,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
0946     */ { 00,   02,   10,   0b0100101011010000 }, /*   29   30   29   29   30   29   30   29   30   30   29   30        354
0947     */ { 07,   01,   30,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
0948     */ { 00,   02,   18,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
0949     */ { 00,   02,   06,   0b1011001001100000 }, /*   30   29   30   30   29   29   30   29   29   30   30   29        354
0950     */ { 05,   01,   26,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
0951     */ { 00,   02,   14,   0b1011001101110000 }, /*   30   29   30   30   29   29   30   30   29   30   30   30        356
0953     */ { 00,   01,   05,   0b1010101011010000 }, /*   30   29   30   29   30   29   30   29   30   30   29   30        355
0953     */ { 01,   01,   23,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30   29   384
0954     */ { 00,   02,   11,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
0955     */ { 09,   02,   01,   0b0100101010111000 }, /*   29   30   29   29   30   29   30   29   30   29   30   30   30   384
0956     */ { 00,   02,   20,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
0957     */ { 00,   02,   08,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
0958     */ { 07,   01,   28,   0b1010101010011000 }, /*   30   29   30   29   30   29   30   29   30   29   29   30   30   384
0959     */ { 00,   02,   16,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
0960     */ { 00,   02,   05,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
0961     */ { 03,   01,   25,   0b0100110110101000 }, /*   29   30   29   29   30   30   29   30   30   29   30   29   30   384
0962     */ { 00,   02,   13,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
0963     */ { 12,   02,   02,   0b1001010101110000 }, /*   30   29   29   30   29   30   29   30   29   30   30   30   29   384
0964     */ { 00,   02,   21,   0b1010001101110000 }, /*   30   29   30   29   29   29   30   30   29   30   30   30        355
0965     */ { 00,   02,   10,   0b0101000101110000 }, /*   29   30   29   30   29   29   29   30   29   30   30   30        354
0966     */ { 08,   01,   30,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   29   383
0967     */ { 00,   02,   17,   0b1101010010110000 }, /*   30   30   29   30   29   30   29   29   30   29   30   30        355
0968     */ { 00,   02,   07,   0b0101101010010000 }, /*   29   30   29   30   30   29   30   29   30   29   29   30        354
0969     */ { 05,   01,   26,   0b0110101101010000 }, /*   29   30   30   29   30   29   30   30   29   30   29   30   29   384
0970     */ { 00,   02,   14,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
0971     */ { 00,   02,   04,   0b0010101011100000 }, /*   29   29   30   29   30   29   30   29   30   30   30   29        354
0972     */ { 02,   01,   24,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30   29   384
0973     */ { 00,   02,   11,   0b1010001011100000 }, /*   30   29   30   29   29   29   30   29   30   30   30   29        354
0974     */ { 10,   01,   31,   0b1100100101101000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29   30   384
0975     */ { 00,   02,   19,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
0976     */ { 00,   02,   08,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
0977     */ { 07,   01,   27,   0b1101101010010000 }, /*   30   30   29   30   30   29   30   29   30   29   29   30   29   384
0978     */ { 00,   02,   15,   0b1011010110100000 }, /*   30   29   30   30   29   30   29   30   30   29   30   29        355
0979     */ { 00,   02,   05,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
0980     */ { 03,   01,   26,   0b0010101011011000 }, /*   29   29   30   29   30   29   30   29   30   30   29   30   30   384
0981     */ { 00,   02,   13,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
0982     */ { 12,   02,   02,   0b1001001011011000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30   30   384
0983     */ { 00,   02,   21,   0b1001001010110000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30        354
0984     */ { 00,   02,   10,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
0985     */ { 09,   01,   29,   0b1011010010101000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29   30   384
0986     */ { 00,   02,   17,   0b1010110010100000 }, /*   30   29   30   29   30   30   29   29   30   29   30   29        354
0987     */ { 00,   02,   06,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
0988     */ { 05,   01,   27,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30   29   384
0989     */ { 00,   02,   14,   0b0100101110110000 }, /*   29   30   29   29   30   29   30   30   30   29   30   30        355
0990     */ { 00,   02,   04,   0b0010010110110000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30        354
0991     */ { 02,   01,   24,   0b1001001010111000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30   30   384
0992     */ { 00,   02,   12,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
0993     */ { 10,   01,   31,   0b0110100101011000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30   30   384
0994     */ { 00,   02,   19,   0b0101100101010000 }, /*   29   30   29   30   30   29   29   30   29   30   29   30        354
0995     */ { 00,   02,   08,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
0996     */ { 07,   01,   28,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30   29   384
0997     */ { 00,   02,   15,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
0998     */ { 00,   02,   05,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
0999     */ { 03,   01,   25,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
1000     */ { 00,   02,   13,   0b0010010101110000 }, /*   29   29   30   29   29   30   29   30   29   30   30   30        354
1001     */ { 12,   02,   03,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   29   383
1002     */ { 00,   02,   21,   0b1101001010100000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29        354
1003     */ { 00,   02,   10,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1004     */ { 09,   01,   31,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1005     */ { 00,   02,   18,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1006     */ { 00,   02,   07,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30        355
1007     */ { 05,   01,   28,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1008     */ { 00,   02,   16,   0b0100101011010000 }, /*   29   30   29   29   30   29   30   29   30   30   29   30        354
1009     */ { 00,   02,   04,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1010     */ { 02,   01,   24,   0b1101001001101000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29   30   384
1011     */ { 00,   02,   12,   0b1011001001100000 }, /*   30   29   30   30   29   29   30   29   29   30   30   29        354
1012     */ { 10,   02,   01,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1013     */ { 00,   02,   19,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1014     */ { 00,   02,   09,   0b0011010110100000 }, /*   29   29   30   30   29   30   29   30   30   29   30   29        354
1015     */ { 06,   01,   29,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30   29   384
1016     */ { 00,   02,   17,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1017     */ { 00,   02,   06,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1018     */ { 04,   01,   26,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
1019     */ { 00,   02,   14,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1020     */ { 12,   02,   03,   0b1010101001011000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30   30   384
1021     */ { 00,   02,   21,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1022     */ { 00,   02,   10,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1023     */ { 09,   01,   31,   0b0010110110101000 }, /*   29   29   30   29   30   30   29   30   30   29   30   29   30   384
1024     */ { 00,   02,   19,   0b0010101101010000 }, /*   29   29   30   29   30   29   30   30   29   30   29   30        354
1025     */ { 00,   02,   07,   0b1001010101110000 }, /*   30   29   29   30   29   30   29   30   29   30   30   30        355
1026     */ { 05,   01,   28,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30   29   383
1027     */ { 00,   02,   15,   0b1100100101110000 }, /*   30   30   29   29   30   29   29   30   29   30   30   30        355
1028     */ { 00,   02,   05,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1029     */ { 02,   01,   24,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   29   383
1030     */ { 00,   02,   11,   0b1101101010010000 }, /*   30   30   29   30   30   29   30   29   30   29   29   30        355
1031     */ { 10,   02,   01,   0b0110101101010000 }, /*   29   30   30   29   30   29   30   30   29   30   29   30   29   384
1032     */ { 00,   02,   20,   0b0110011011010000 }, /*   29   30   30   29   29   30   30   29   30   30   29   30        355
1033     */ { 00,   02,   09,   0b0010011011100000 }, /*   29   29   30   29   29   30   30   29   30   30   30   29        354
1034     */ { 06,   01,   29,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30   29   384
1035     */ { 00,   02,   17,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1036     */ { 00,   02,   06,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1037     */ { 04,   01,   25,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1038     */ { 00,   02,   13,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1039     */ { 12,   02,   02,   0b1101011010010000 }, /*   30   30   29   30   29   30   30   29   30   29   29   30   29   384
1040     */ { 00,   02,   21,   0b1011010110000000 }, /*   30   29   30   30   29   30   29   30   30   29   29   29        354
1041     */ { 00,   02,   09,   0b1101011010110000 }, /*   30   30   29   30   29   30   30   29   30   29   30   30        356
1042     */ { 09,   01,   31,   0b0010011011011000 }, /*   29   29   30   29   29   30   30   29   30   30   29   30   30   384
1043     */ { 00,   02,   19,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1044     */ { 00,   02,   08,   0b1001001010110000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30        354
1045     */ { 05,   01,   27,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1046     */ { 00,   02,   15,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1047     */ { 00,   02,   04,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1048     */ { 01,   01,   24,   0b1011010101011000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   30   385
1049     */ { 00,   02,   12,   0b0010110101010000 }, /*   29   29   30   29   30   30   29   30   29   30   29   30        354
1050     */ { 11,   02,   01,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30   29   384
1051     */ { 00,   02,   20,   0b0100101110110000 }, /*   29   30   29   29   30   29   30   30   30   29   30   30        355
1052     */ { 00,   02,   10,   0b0010010110110000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30        354
1053     */ { 07,   01,   29,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
1054     */ { 00,   02,   17,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1055     */ { 00,   02,   06,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1056     */ { 03,   01,   26,   0b0110101010101000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29   30   384
1057     */ { 00,   02,   13,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1058     */ { 12,   02,   02,   0b1010101101011000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30   30   385
1059     */ { 00,   02,   22,   0b0010011101010000 }, /*   29   29   30   29   29   30   30   30   29   30   29   30        354
1060     */ { 00,   02,   11,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1061     */ { 08,   01,   30,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
1062     */ { 00,   02,   18,   0b1010010101100000 }, /*   30   29   30   29   29   30   29   30   29   30   30   29        354
1063     */ { 00,   02,   07,   0b1101001001100000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29        354
1064     */ { 05,   01,   27,   0b1110100100110000 }, /*   30   30   30   29   30   29   29   30   29   29   30   30   29   384
1065     */ { 00,   02,   14,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1066     */ { 00,   02,   04,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1067     */ { 01,   01,   24,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30   29   384
1068     */ { 00,   02,   12,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30        355
1069     */ { 11,   02,   01,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1070     */ { 00,   02,   20,   0b0100100111010000 }, /*   29   30   29   29   30   29   29   30   30   30   29   30        354
1071     */ { 00,   02,   09,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1072     */ { 07,   01,   29,   0b1101001001101000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29   30   384
1073     */ { 00,   02,   16,   0b1010101001100000 }, /*   30   29   30   29   30   29   30   29   29   30   30   29        354
1074     */ { 00,   02,   05,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1075     */ { 04,   01,   26,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1076     */ { 00,   02,   14,   0b0011010110100000 }, /*   29   29   30   30   29   30   29   30   30   29   30   29        354
1077     */ { 00,   02,   02,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1078     */ { 01,   01,   23,   0b0100101110011000 }, /*   29   30   29   29   30   29   30   30   30   29   29   30   30   384
1079     */ { 00,   02,   11,   0b0100010110110000 }, /*   29   30   29   29   29   30   29   30   30   29   30   30        354
1080     */ { 09,   01,   31,   0b1010010010111000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30   30   384
1081     */ { 00,   02,   18,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1082     */ { 00,   02,   07,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30        354
1083     */ { 06,   01,   27,   0b1011010101001000 }, /*   30   29   30   30   29   30   29   30   29   30   29   29   30   384
1084     */ { 00,   02,   15,   0b0110101101010000 }, /*   29   30   30   29   30   29   30   30   29   30   29   30        355
1085     */ { 00,   02,   04,   0b0010110110100000 }, /*   29   29   30   29   30   30   29   30   30   29   30   29        354
1086     */ { 02,   01,   24,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30   29   384
1087     */ { 00,   02,   12,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1088     */ { 12,   02,   02,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30   29   383
1089     */ { 00,   02,   19,   0b1100100101110000 }, /*   30   30   29   29   30   29   29   30   29   30   30   30        355
1090     */ { 00,   02,   09,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1091     */ { 08,   01,   29,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   29   383
1092     */ { 00,   02,   16,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30        355
1093     */ { 00,   02,   05,   0b0101101101000000 }, /*   29   30   29   30   30   29   30   30   29   30   29   29        354
1094     */ { 04,   01,   25,   0b1010101101101000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29   30   385
1095     */ { 00,   02,   14,   0b0010101011100000 }, /*   29   29   30   29   30   29   30   29   30   30   30   29        354
1096     */ { 00,   02,   03,   0b1110011000010000 }, /*   30   30   30   29   29   30   30   29   29   29   29   30        354
1097     */ { 02,   01,   22,   0b1100100101110000 }, /*   30   30   29   29   30   29   29   30   29   30   30   30   29   384
1098     */ { 00,   02,   10,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1099     */ { 09,   01,   30,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1100     */ { 00,   02,   18,   0b0101010010100000 }, /*   29   30   29   30   29   30   29   29   30   29   30   29        353
1101     */ { 00,   02,   07,   0b1101011001010000 }, /*   30   30   29   30   29   30   30   29   29   30   29   30        355
1102     */ { 06,   01,   28,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1103     */ { 00,   02,   16,   0b0101010111010000 }, /*   29   30   29   30   29   30   29   30   30   30   29   30        355
1104     */ { 00,   02,   06,   0b0010011011010000 }, /*   29   29   30   29   29   30   30   29   30   30   29   30        354
1105     */ { 02,   01,   25,   0b1001001011101000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29   30   384
1106     */ { 00,   02,   13,   0b1001001010110000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30        354
1107     */ { 10,   02,   02,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1108     */ { 00,   02,   21,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1109     */ { 00,   02,   09,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1110     */ { 08,   01,   29,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1111     */ { 00,   02,   17,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1112     */ { 00,   02,   07,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
1113     */ { 04,   01,   27,   0b0010010110111000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30   30   384
1114     */ { 00,   02,   15,   0b0100010101110000 }, /*   29   30   29   29   29   30   29   30   29   30   30   30        354
1115     */ { 00,   02,   04,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1116     */ { 01,   01,   24,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1117     */ { 00,   02,   11,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1118     */ { 09,   01,   31,   0b0111001010101000 }, /*   29   30   30   30   29   29   30   29   30   29   30   29   30   384
1119     */ { 00,   02,   19,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1120     */ { 00,   02,   08,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1121     */ { 05,   01,   28,   0b0100101101101000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29   30   384
1122     */ { 00,   02,   16,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1123     */ { 00,   02,   05,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1124     */ { 03,   01,   26,   0b0101001001110000 }, /*   29   30   29   30   29   29   30   29   29   30   30   30   29   383
1125     */ { 00,   02,   12,   0b1101001001100000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29        354
1126     */ { 11,   02,   01,   0b1110100100110000 }, /*   30   30   30   29   30   29   29   30   29   29   30   30   29   384
1127     */ { 00,   02,   20,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1128     */ { 00,   02,   10,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1129     */ { 08,   01,   29,   0b1001101110010000 }, /*   30   29   29   30   30   29   30   30   30   29   29   30   29   384
1130     */ { 00,   02,   17,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30        355
1131     */ { 00,   02,   07,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1132     */ { 04,   01,   27,   0b1010010011101000 }, /*   30   29   30   29   29   30   29   29   30   30   30   29   30   384
1133     */ { 00,   02,   14,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1134     */ { 00,   02,   03,   0b1101001001010000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30        354
1135     */ { 02,   01,   23,   0b1101100100101000 }, /*   30   30   29   30   30   29   29   30   29   29   30   29   30   384
1136     */ { 00,   02,   11,   0b1011010101000000 }, /*   30   29   30   30   29   30   29   30   29   30   29   29        354
1137     */ { 10,   01,   30,   0b1101011010101000 }, /*   30   30   29   30   29   30   30   29   30   29   30   29   30   385
1138     */ { 00,   02,   19,   0b0010110110100000 }, /*   29   29   30   29   30   30   29   30   30   29   30   29        354
1139     */ { 00,   02,   08,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1140     */ { 06,   01,   29,   0b0100101011011000 }, /*   29   30   29   29   30   29   30   29   30   30   29   30   30   384
1141     */ { 00,   02,   16,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1142     */ { 00,   02,   05,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1143     */ { 04,   01,   25,   0b1011001001011000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30   30   384
1144     */ { 00,   02,   13,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30        354
1145     */ { 11,   02,   01,   0b1011010100101000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29   30   384
1146     */ { 00,   02,   20,   0b0110101101000000 }, /*   29   30   30   29   30   29   30   30   29   30   29   29        354
1147     */ { 00,   02,   09,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1148     */ { 08,   01,   30,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30   29   384
1149     */ { 00,   02,   17,   0b1001100101110000 }, /*   30   29   29   30   30   29   29   30   29   30   30   30        355
1150     */ { 00,   02,   07,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1151     */ { 04,   01,   27,   0b0110010010111000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   30   384
1152     */ { 00,   02,   15,   0b0101010010110000 }, /*   29   30   29   30   29   30   29   29   30   29   30   30        354
1153     */ { 12,   02,   03,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   29   383
1154     */ { 00,   02,   21,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30        355
1155     */ { 00,   02,   11,   0b0101101011000000 }, /*   29   30   29   30   30   29   30   29   30   30   29   29        354
1156     */ { 10,   01,   31,   0b1010101101101000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29   30   385
1157     */ { 00,   02,   19,   0b0010011011100000 }, /*   29   29   30   29   29   30   30   29   30   30   30   29        354
1158     */ { 00,   02,   08,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1159     */ { 06,   01,   28,   0b1100100101110000 }, /*   30   30   29   29   30   29   29   30   29   30   30   30   29   384
1160     */ { 00,   02,   16,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1161     */ { 00,   02,   04,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1162     */ { 02,   01,   24,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30   29   384
1163     */ { 00,   02,   12,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1164     */ { 11,   02,   02,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1165     */ { 00,   02,   20,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
1166     */ { 00,   02,   10,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1167     */ { 07,   01,   30,   0b1001001011101000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29   30   384
1168     */ { 00,   02,   18,   0b1001001001110000 }, /*   30   29   29   30   29   29   30   29   29   30   30   30        354
1169     */ { 00,   02,   06,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1170     */ { 05,   01,   26,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1171     */ { 00,   02,   14,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1172     */ { 00,   02,   03,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1173     */ { 01,   01,   23,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1174     */ { 00,   02,   11,   0b0100110110110000 }, /*   29   30   29   29   30   30   29   30   30   29   30   30        355
1175     */ { 09,   02,   01,   0b0010010110110000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30   29   383
1176     */ { 00,   02,   19,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1177     */ { 00,   02,   08,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1178     */ { 06,   01,   28,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1179     */ { 00,   02,   16,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1180     */ { 00,   02,   05,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1181     */ { 03,   01,   24,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1182     */ { 00,   02,   12,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1183     */ { 11,   02,   02,   0b0100101101101000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29   30   384
1184     */ { 00,   02,   21,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1185     */ { 00,   02,   09,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1186     */ { 07,   01,   30,   0b0101001001110000 }, /*   29   30   29   30   29   29   30   29   29   30   30   30   29   383
1187     */ { 00,   02,   17,   0b1101000010010000 }, /*   30   30   29   30   29   29   29   29   30   29   29   30        353
1188     */ { 00,   01,   08,   0b0111010010011000 }, /*   29   30   30   30   29   30   29   29   30   29   29   30        354
1189     */ { 05,   01,   26,   0b0110101010101000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29   30   384
1190     */ { 00,   02,   14,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1191     */ { 00,   02,   03,   0b1001101101010000 }, /*   30   29   29   30   30   29   30   30   29   30   29   30        355
1192     */ { 02,   01,   24,   0b0100101101101000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29   30   384
1193     */ { 00,   02,   11,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1194     */ { 10,   01,   31,   0b1010010011101000 }, /*   30   29   30   29   29   30   29   29   30   30   30   29   30   384
1195     */ { 00,   02,   19,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1196     */ { 00,   02,   08,   0b1101001001100000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29        354
1197     */ { 06,   01,   27,   0b1101010100101000 }, /*   30   30   29   30   29   30   29   30   29   29   30   29   30   384
1198     */ { 00,   02,   15,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1199     */ { 00,   02,   04,   0b1101011010100000 }, /*   30   30   29   30   29   30   30   29   30   29   30   29        355
1200     */ { 02,   01,   25,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30   29   384
1201     */ { 00,   02,   12,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1202     */ { 12,   02,   02,   0b0100100111011000 }, /*   29   30   29   29   30   29   29   30   30   30   29   30   30   384
1203     */ { 00,   02,   21,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1204     */ { 00,   02,   10,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1205     */ { 08,   01,   29,   0b1010101001011000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30   30   384
1206     */ { 00,   02,   17,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30        354
1207     */ { 00,   02,   06,   0b1011010101000000 }, /*   30   29   30   30   29   30   29   30   29   30   29   29        354
1208     */ { 04,   01,   26,   0b1011010110100000 }, /*   30   29   30   30   29   30   29   30   30   29   30   29   29   384
1209     */ { 00,   02,   13,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1210     */ { 00,   02,   03,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1211     */ { 02,   01,   24,   0b0100100110111000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30   30   384
1212     */ { 00,   02,   12,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1213     */ { 09,   01,   31,   0b0110010010111000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   30   384
1214     */ { 00,   02,   19,   0b0101010010110000 }, /*   29   30   29   30   29   30   29   29   30   29   30   30        354
1215     */ { 00,   02,   08,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30        354
1216     */ { 07,   01,   28,   0b0110110100101000 }, /*   29   30   30   29   30   30   29   30   29   29   30   29   30   384
1217     */ { 00,   02,   15,   0b0101101011000000 }, /*   29   30   29   30   30   29   30   29   30   30   29   29        354
1218     */ { 00,   02,   04,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1219     */ { 03,   01,   25,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30   29   384
1220     */ { 00,   02,   13,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1221     */ { 12,   02,   01,   0b1100100101110000 }, /*   30   30   29   29   30   29   29   30   29   30   30   30   29   384
1222     */ { 00,   02,   20,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1223     */ { 00,   02,   09,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1224     */ { 08,   01,   29,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30   29   384
1225     */ { 00,   02,   16,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1226     */ { 00,   02,   06,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1227     */ { 05,   01,   26,   0b1010101011011000 }, /*   30   29   30   29   30   29   30   29   30   30   29   30   30   385
1228     */ { 00,   02,   15,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1229     */ { 00,   02,   03,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1230     */ { 02,   01,   23,   0b1100100101011000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30   30   384
1231     */ { 00,   02,   11,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1232     */ { 09,   01,   31,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1233     */ { 00,   02,   18,   0b1011001010100000 }, /*   30   29   30   30   29   29   30   29   30   29   30   29        354
1234     */ { 00,   02,   07,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1235     */ { 07,   01,   28,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1236     */ { 00,   02,   16,   0b0100110110100000 }, /*   29   30   29   29   30   30   29   30   30   29   30   29        354
1237     */ { 00,   02,   04,   0b1010010110110000 }, /*   30   29   30   29   29   30   29   30   30   29   30   30        355
1238     */ { 04,   01,   25,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
1239     */ { 00,   02,   13,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1240     */ { 12,   02,   02,   0b1010100100111000 }, /*   30   29   30   29   30   29   29   30   29   29   30   30   30   384
1241     */ { 00,   02,   20,   0b0110100100110000 }, /*   29   30   30   29   30   29   29   30   29   29   30   30        354
1242     */ { 00,   02,   09,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1243     */ { 08,   01,   29,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1244     */ { 00,   02,   17,   0b1010111001010000 }, /*   30   29   30   29   30   30   30   29   29   30   29   30        355
1245     */ { 00,   02,   06,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1246     */ { 04,   01,   26,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
1247     */ { 00,   02,   14,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1248     */ { 00,   02,   04,   0b0101001001110000 }, /*   29   30   29   30   29   29   30   29   29   30   30   30        354
1249     */ { 02,   01,   23,   0b0110100100110000 }, /*   29   30   30   29   30   29   29   30   29   29   30   30   29   383
1250     */ { 00,   02,   10,   0b1110010100110000 }, /*   30   30   30   29   29   30   29   30   29   29   30   30        355
1251     */ { 10,   01,   31,   0b0110110010011000 }, /*   29   30   30   29   30   30   29   29   30   29   29   30   30   384
1252     */ { 00,   02,   19,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1253     */ { 00,   02,   07,   0b0101101011010000 }, /*   29   30   29   30   30   29   30   29   30   30   29   30        355
1254     */ { 06,   01,   28,   0b0100101101101000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29   30   384
1255     */ { 00,   02,   16,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1256     */ { 00,   02,   05,   0b1010010011100000 }, /*   30   29   30   29   29   30   29   29   30   30   30   29        354
1257     */ { 04,   01,   24,   0b1101001001101000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29   30   384
1258     */ { 00,   02,   12,   0b1101001001100000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29        354
1259     */ { 11,   02,   01,   0b1101010100101000 }, /*   30   30   29   30   29   30   29   30   29   29   30   29   30   384
1260     */ { 00,   02,   20,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1261     */ { 00,   02,   08,   0b1011011010100000 }, /*   30   29   30   30   29   30   30   29   30   29   30   29        355
1262     */ { 09,   01,   29,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30   29   384
1263     */ { 00,   02,   17,   0b0101010101110000 }, /*   29   30   29   30   29   30   29   30   29   30   30   30        355
1264     */ { 00,   02,   07,   0b0100100111010000 }, /*   29   30   29   29   30   29   29   30   30   30   29   30        354
1265     */ { 05,   01,   26,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
1266     */ { 00,   02,   14,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1267     */ { 00,   02,   03,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30        354
1268     */ { 01,   01,   23,   0b1011010100101000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29   30   384
1269     */ { 00,   02,   10,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1270     */ { 11,   01,   30,   0b1011010111000000 }, /*   30   29   30   30   29   30   29   30   30   30   29   29   29   384
1271     */ { 00,   02,   18,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1272     */ { 00,   02,   08,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1273     */ { 06,   01,   28,   0b0100100110111000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30   30   384
1274     */ { 00,   02,   16,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1275     */ { 00,   02,   05,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1276     */ { 03,   01,   25,   0b0110101001011000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   30   384
1277     */ { 00,   02,   12,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30        354
1278     */ { 11,   02,   01,   0b0110101100101000 }, /*   29   30   30   29   30   29   30   30   29   29   30   29   30   384
1279     */ { 00,   02,   20,   0b0101101011000000 }, /*   29   30   29   30   30   29   30   29   30   30   29   29        354
1280     */ { 00,   02,   09,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1281     */ { 08,   01,   29,   0b0010101011101000 }, /*   29   29   30   29   30   29   30   29   30   30   30   29   30   384
1282     */ { 00,   02,   17,   0b0100100111100000 }, /*   29   30   29   29   30   29   29   30   30   30   30   29        354
1283     */ { 00,   02,   06,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1284     */ { 05,   01,   26,   0b1101001001011000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30   30   384
1285     */ { 00,   02,   13,   0b1011001001010000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30        354
1286     */ { 00,   02,   02,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1287     */ { 02,   01,   22,   0b1111001010010000 }, /*   30   30   30   30   29   29   30   29   30   29   29   30   29   384
1288     */ { 00,   02,   10,   0b1011010110100000 }, /*   30   29   30   30   29   30   29   30   30   29   30   29        355
1289     */ { 10,   01,   30,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30   29   384
1290     */ { 00,   02,   18,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1291     */ { 00,   02,   08,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1292     */ { 06,   01,   28,   0b1010010010111000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30   30   384
1293     */ { 00,   02,   15,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1294     */ { 00,   02,   04,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30        354
1295     */ { 04,   01,   24,   0b1011010100101000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29   30   384
1296     */ { 00,   02,   12,   0b0110110101000000 }, /*   29   30   30   29   30   30   29   30   29   30   29   29        354
1297     */ { 12,   01,   31,   0b1010110101100000 }, /*   30   29   30   29   30   30   29   30   29   30   30   29   29   384
1298     */ { 00,   02,   19,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1299     */ { 00,   02,   09,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1300     */ { 08,   01,   30,   0b0000100101111000 }, /*   29   29   29   29   30   29   29   30   29   30   30   30   30   383
1301     */ { 00,   02,   18,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1302     */ { 00,   02,   07,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1303     */ { 05,   01,   27,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   29   383
1304     */ { 00,   02,   14,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30        355
1305     */ { 00,   02,   03,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1306     */ { 01,   01,   23,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29   29   384
1307     */ { 00,   02,   11,   0b1010011011100000 }, /*   30   29   30   29   29   30   30   29   30   30   30   29        355
1308     */ { 11,   02,   01,   0b1001001011101000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29   30   384
1309     */ { 00,   02,   19,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1310     */ { 00,   02,   08,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1311     */ { 07,   01,   28,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1312     */ { 00,   02,   16,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1313     */ { 00,   02,   04,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1314     */ { 03,   01,   25,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1315     */ { 00,   02,   13,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1316     */ { 00,   02,   02,   0b1010011011010000 }, /*   30   29   30   29   29   30   30   29   30   30   29   30        355
1317     */ { 01,   01,   22,   0b1001001011101000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29   30   384
1318     */ { 00,   02,   10,   0b1001001010110000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30        354
1319     */ { 08,   01,   30,   0b1010010101011000 }, /*   30   29   30   29   29   30   29   30   29   30   29   30   30   384
1320     */ { 00,   02,   18,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1321     */ { 00,   02,   06,   0b1011001010100000 }, /*   30   29   30   30   29   29   30   29   30   29   30   29        354
1322     */ { 05,   01,   26,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1323     */ { 00,   02,   14,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1324     */ { 00,   02,   04,   0b0100110110100000 }, /*   29   30   29   29   30   30   29   30   30   29   30   29        354
1325     */ { 01,   01,   23,   0b1010010111010000 }, /*   30   29   30   29   29   30   29   30   30   30   29   30   29   384
1326     */ { 00,   02,   11,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1327     */ { 09,   02,   01,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
1328     */ { 00,   02,   20,   0b0101001001110000 }, /*   29   30   29   30   29   29   30   29   29   30   30   30        354
1329     */ { 00,   02,   08,   0b0110100100110000 }, /*   29   30   30   29   30   29   29   30   29   29   30   30        354
1330     */ { 07,   01,   28,   0b0110101010011000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30   30   384
1331     */ { 00,   02,   16,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1332     */ { 00,   02,   05,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1333     */ { 03,   01,   25,   0b0100101110101000 }, /*   29   30   29   29   30   29   30   30   30   29   30   29   30   384
1334     */ { 00,   02,   13,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1335     */ { 12,   02,   02,   0b1010011001110000 }, /*   30   29   30   29   29   30   30   29   29   30   30   30   29   384
1336     */ { 00,   02,   21,   0b1010001011100000 }, /*   30   29   30   29   29   29   30   29   30   30   30   29        354
1337     */ { 00,   02,   09,   0b1101000101100000 }, /*   30   30   29   30   29   29   29   30   29   30   30   29        354
1338     */ { 08,   01,   29,   0b1110100100110000 }, /*   30   30   30   29   30   29   29   30   29   29   30   30   29   384
1339     */ { 00,   02,   17,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1340     */ { 00,   02,   06,   0b1101101010100000 }, /*   30   30   29   30   30   29   30   29   30   29   30   29        355
1341     */ { 05,   01,   26,   0b0101101101010000 }, /*   29   30   29   30   30   29   30   30   29   30   29   30   29   384
1342     */ { 00,   02,   14,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1343     */ { 00,   02,   04,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1344     */ { 02,   01,   24,   0b1010001011101000 }, /*   30   29   30   29   29   29   30   29   30   30   30   29   30   384
1345     */ { 00,   02,   11,   0b1010001011010000 }, /*   30   29   30   29   29   29   30   29   30   30   29   30        354
1346     */ { 10,   01,   31,   0b1101000101011000 }, /*   30   30   29   30   29   29   29   30   29   30   29   30   30   384
1347     */ { 00,   02,   19,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30        354
1348     */ { 00,   02,   08,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1349     */ { 07,   01,   27,   0b1101011010100000 }, /*   30   30   29   30   29   30   30   29   30   29   30   29   29   384
1350     */ { 00,   02,   15,   0b1010110110100000 }, /*   30   29   30   29   30   30   29   30   30   29   30   29        355
1351     */ { 00,   02,   05,   0b0101010111010000 }, /*   29   30   29   30   29   30   29   30   30   30   29   30        355
1352     */ { 03,   01,   26,   0b0100100111011000 }, /*   29   30   29   29   30   29   29   30   30   30   29   30   30   384
1353     */ { 00,   02,   13,   0b0100010110110000 }, /*   29   30   29   29   29   30   29   30   30   29   30   30        354
1354     */ { 00,   02,   02,   0b1010001010110000 }, /*   30   29   30   29   29   29   30   29   30   29   30   30        354
1355     */ { 01,   01,   22,   0b1101000101011000 }, /*   30   30   29   30   29   29   29   30   29   30   29   30   30   384
1356     */ { 00,   02,   10,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30        354
1357     */ { 09,   01,   29,   0b1011010100101000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29   30   384
1358     */ { 00,   02,   17,   0b0110101100100000 }, /*   29   30   30   29   30   29   30   30   29   29   30   29        354
1359     */ { 00,   02,   06,   0b1010110101100000 }, /*   30   29   30   29   30   30   29   30   29   30   30   29        355
1360     */ { 05,   01,   27,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30   29   384
1361     */ { 00,   02,   14,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1362     */ { 00,   02,   04,   0b0100010101110000 }, /*   29   30   29   29   29   30   29   30   29   30   30   30        354
1363     */ { 03,   01,   24,   0b1010001010111000 }, /*   30   29   30   29   29   29   30   29   30   29   30   30   30   384
1364     */ { 00,   02,   12,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1365     */ { 10,   01,   31,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30   29   383
1366     */ { 00,   02,   18,   0b1101100101010000 }, /*   30   30   29   30   30   29   29   30   29   30   29   30        355
1367     */ { 00,   02,   08,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1368     */ { 07,   01,   28,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29   29   384
1369     */ { 00,   02,   15,   0b1010011011100000 }, /*   30   29   30   29   29   30   30   29   30   30   30   29        355
1370     */ { 00,   02,   05,   0b0101001011100000 }, /*   29   30   29   30   29   29   30   29   30   30   30   29        354
1371     */ { 03,   01,   25,   0b1100010101110000 }, /*   30   30   29   29   29   30   29   30   29   30   30   30   29   384
1372     */ { 00,   02,   13,   0b1010010101100000 }, /*   30   29   30   29   29   30   29   30   29   30   30   29        354
1373     */ { 11,   02,   01,   0b1101001010101000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29   30   384
1374     */ { 00,   02,   20,   0b1101001010100000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29        354
1375     */ { 00,   02,   09,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1376     */ { 09,   01,   30,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1377     */ { 00,   02,   17,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1378     */ { 00,   02,   06,   0b1010011011010000 }, /*   30   29   30   29   29   30   30   29   30   30   29   30        355
1379     */ { 05,   01,   27,   0b0101001011101000 }, /*   29   30   29   30   29   29   30   29   30   30   30   29   30   384
1380     */ { 00,   02,   15,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1381     */ { 00,   02,   03,   0b1010100011010000 }, /*   30   29   30   29   30   29   29   29   30   30   29   30        354
1382     */ { 02,   01,   23,   0b1101001010101000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29   30   384
1383     */ { 00,   02,   11,   0b1011001010100000 }, /*   30   29   30   30   29   29   30   29   30   29   30   29        354
1384     */ { 10,   01,   31,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1385     */ { 00,   02,   18,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1386     */ { 00,   02,   08,   0b0100110110100000 }, /*   29   30   29   29   30   30   29   30   30   29   30   29        354
1387     */ { 06,   01,   28,   0b1010010111010000 }, /*   30   29   30   29   29   30   29   30   30   30   29   30   29   384
1388     */ { 00,   02,   16,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1389     */ { 00,   02,   05,   0b0101000110110000 }, /*   29   30   29   30   29   29   29   30   30   29   30   30        354
1390     */ { 04,   01,   25,   0b1010100010111000 }, /*   30   29   30   29   30   29   29   29   30   29   30   30   30   384
1391     */ { 00,   02,   13,   0b0110010100110000 }, /*   29   30   30   29   29   30   29   30   29   29   30   30        354
1392     */ { 12,   02,   02,   0b0110101010011000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30   30   384
1393     */ { 00,   02,   20,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1394     */ { 00,   02,   09,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1395     */ { 09,   01,   30,   0b0010101110101000 }, /*   29   29   30   29   30   29   30   30   30   29   30   29   30   384
1396     */ { 00,   02,   18,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1397     */ { 00,   02,   06,   0b1100001101110000 }, /*   30   30   29   29   29   29   30   30   29   30   30   30        355
1398     */ { 05,   01,   27,   0b0101000101110000 }, /*   29   30   29   30   29   29   29   30   29   30   30   30   29   383
1399     */ { 00,   02,   14,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1400     */ { 00,   02,   03,   0b1110010010110000 }, /*   30   30   30   29   29   30   29   29   30   29   30   30        355
1401     */ { 03,   01,   24,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30   29   383
1402     */ { 00,   02,   11,   0b1101101010100000 }, /*   30   30   29   30   30   29   30   29   30   29   30   29        355
1403     */ { 11,   02,   01,   0b0101101101010000 }, /*   29   30   29   30   30   29   30   30   29   30   29   30   29   384
1404     */ { 00,   02,   20,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1405     */ { 00,   02,   09,   0b0010101011100000 }, /*   29   29   30   29   30   29   30   29   30   30   30   29        354
1406     */ { 07,   01,   29,   0b1010001011101000 }, /*   30   29   30   29   29   29   30   29   30   30   30   29   30   384
1407     */ { 00,   02,   17,   0b1010001011010000 }, /*   30   29   30   29   29   29   30   29   30   30   29   30        354
1408     */ { 00,   02,   06,   0b1101000101010000 }, /*   30   30   29   30   29   29   29   30   29   30   29   30        354
1409     */ { 04,   01,   25,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1410     */ { 00,   02,   13,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1411     */ { 12,   02,   02,   0b1011011010010000 }, /*   30   29   30   30   29   30   30   29   30   29   29   30   29   384
1412     */ { 00,   02,   21,   0b1010110110100000 }, /*   30   29   30   29   30   30   29   30   30   29   30   29        355
1413     */ { 00,   02,   10,   0b0101010111010000 }, /*   29   30   29   30   29   30   29   30   30   30   29   30        355
1414     */ { 09,   01,   31,   0b0010010111011000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30   30   384
1415     */ { 00,   02,   19,   0b0100010110110000 }, /*   29   30   29   29   29   30   29   30   30   29   30   30        354
1416     */ { 00,   02,   08,   0b1010001010110000 }, /*   30   29   30   29   29   29   30   29   30   29   30   30        354
1417     */ { 05,   01,   27,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1418     */ { 00,   02,   15,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1419     */ { 00,   02,   04,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1420     */ { 01,   01,   24,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1421     */ { 00,   02,   11,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1422     */ { 12,   02,   01,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30   29   384
1423     */ { 00,   02,   20,   0b0100101101110000 }, /*   29   30   29   29   30   29   30   30   29   30   30   30        355
1424     */ { 00,   02,   10,   0b0100010101110000 }, /*   29   30   29   29   29   30   29   30   29   30   30   30        354
1425     */ { 07,   01,   29,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
1426     */ { 00,   02,   17,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1427     */ { 00,   02,   06,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1428     */ { 04,   01,   26,   0b0110110010101000 }, /*   29   30   30   29   30   30   29   29   30   29   30   29   30   384
1429     */ { 00,   02,   13,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1430     */ { 12,   02,   02,   0b1001101101010000 }, /*   30   29   29   30   30   29   30   30   29   30   29   30   29   384
1431     */ { 00,   02,   21,   0b1010011011100000 }, /*   30   29   30   29   29   30   30   29   30   30   30   29        355
1432     */ { 00,   02,   11,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1433     */ { 08,   01,   30,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
1434     */ { 00,   02,   18,   0b1010010101100000 }, /*   30   29   30   29   29   30   29   30   29   30   30   29        354
1435     */ { 00,   02,   07,   0b1101001010100000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29        354
1436     */ { 06,   01,   27,   0b1110100101010000 }, /*   30   30   30   29   30   29   29   30   29   30   29   30   29   384
1437     */ { 00,   02,   14,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1438     */ { 00,   02,   04,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1439     */ { 02,   01,   24,   0b1010101011010000 }, /*   30   29   30   29   30   29   30   29   30   30   29   30   29   384
1440     */ { 00,   02,   12,   0b1001010111100000 }, /*   30   29   29   30   29   30   29   30   30   30   30   29        355
1441     */ { 11,   02,   01,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1442     */ { 00,   02,   20,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1443     */ { 00,   02,   09,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1444     */ { 07,   01,   29,   0b1101001001110000 }, /*   30   30   29   30   29   29   30   29   29   30   30   30   29   384
1445     */ { 00,   02,   16,   0b1011001010100000 }, /*   30   29   30   30   29   29   30   29   30   29   30   29        354
1446     */ { 00,   02,   05,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1447     */ { 04,   01,   26,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1448     */ { 00,   02,   14,   0b0010110110100000 }, /*   29   29   30   29   30   30   29   30   30   29   30   29        354
1449     */ { 00,   02,   02,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1450     */ { 01,   01,   23,   0b0100101010111000 }, /*   29   30   29   29   30   29   30   29   30   29   30   30   30   384
1451     */ { 00,   02,   11,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1452     */ { 09,   01,   31,   0b1010010010111000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30   30   384
1453     */ { 00,   02,   18,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1454     */ { 00,   02,   07,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30        354
1455     */ { 06,   01,   27,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1456     */ { 00,   02,   15,   0b0110101101010000 }, /*   29   30   30   29   30   29   30   30   29   30   29   30        355
1457     */ { 00,   02,   04,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1458     */ { 02,   01,   24,   0b1001010101110000 }, /*   30   29   29   30   29   30   29   30   29   30   30   30   29   384
1459     */ { 00,   02,   12,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1460     */ { 11,   02,   02,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30   29   383
1461     */ { 00,   02,   19,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1462     */ { 00,   02,   08,   0b1110010010110000 }, /*   30   30   30   29   29   30   29   29   30   29   30   30        355
1463     */ { 07,   01,   29,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30   29   383
1464     */ { 00,   02,   16,   0b1101101010100000 }, /*   30   30   29   30   30   29   30   29   30   29   30   29        355
1465     */ { 00,   02,   05,   0b0101101011010000 }, /*   29   30   29   30   30   29   30   29   30   30   29   30        355
1466     */ { 03,   01,   26,   0b0010101101101000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29   30   384
1467     */ { 00,   02,   14,   0b0010011011100000 }, /*   29   29   30   29   29   30   30   29   30   30   30   29        354
1468     */ { 00,   02,   03,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1469     */ { 02,   01,   22,   0b1100100101101000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29   30   384
1470     */ { 00,   02,   10,   0b1100100101010000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30        354
1471     */ { 09,   01,   30,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1472     */ { 00,   02,   18,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1473     */ { 00,   02,   06,   0b1011011010010000 }, /*   30   29   30   30   29   30   30   29   30   29   29   30        355
1474     */ { 06,   01,   27,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30   29   384
1475     */ { 00,   02,   15,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
1476     */ { 00,   02,   05,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1477     */ { 02,   01,   24,   0b1001001011011000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30   30   384
1478     */ { 00,   02,   12,   0b1001001010110000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30        354
1479     */ { 10,   02,   01,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1480     */ { 00,   02,   20,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1481     */ { 00,   02,   08,   0b0111010010100000 }, /*   29   30   30   30   29   30   29   29   30   29   30   29        354
1482     */ { 08,   01,   28,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1483     */ { 00,   02,   16,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29        355
1484     */ { 00,   02,   06,   0b0101001110110000 }, /*   29   30   29   30   29   29   30   30   30   29   30   30        355
1485     */ { 04,   01,   26,   0b0010010110111000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30   30   384
1486     */ { 00,   02,   14,   0b0010010101110000 }, /*   29   29   30   29   29   30   29   30   29   30   30   30        354
1487     */ { 00,   02,   03,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1488     */ { 01,   01,   23,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1489     */ { 00,   02,   10,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1490     */ { 09,   01,   30,   0b0110101010101000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29   30   384
1491     */ { 00,   02,   18,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1492     */ { 00,   02,   07,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1493     */ { 05,   01,   27,   0b0100101101101000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29   30   384
1494     */ { 00,   02,   15,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1495     */ { 00,   02,   04,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1496     */ { 03,   01,   25,   0b0101001001110000 }, /*   29   30   29   30   29   29   30   29   29   30   30   30   29   383
1497     */ { 00,   02,   11,   0b1101001010100000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29        354
1498     */ { 11,   01,   31,   0b1101100101010000 }, /*   30   30   29   30   30   29   29   30   29   30   29   30   29   384
1499     */ { 00,   02,   19,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1500     */ { 00,   02,   09,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1501     */ { 07,   01,   29,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30   29   384
1502     */ { 00,   02,   17,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1503     */ { 00,   02,   07,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1504     */ { 04,   01,   27,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
1505     */ { 00,   02,   14,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1506     */ { 00,   02,   03,   0b1101001001010000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30        354
1507     */ { 01,   01,   23,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30   29   384
1508     */ { 00,   02,   11,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1509     */ { 09,   01,   31,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1510     */ { 00,   02,   19,   0b0010110110100000 }, /*   29   29   30   29   30   30   29   30   30   29   30   29        354
1511     */ { 00,   02,   08,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1512     */ { 05,   01,   29,   0b0100100110111000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30   30   384
1513     */ { 00,   02,   16,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1514     */ { 00,   02,   05,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1515     */ { 04,   01,   25,   0b1011001001011000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30   30   384
1516     */ { 00,   02,   13,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30        354
1517     */ { 12,   02,   01,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1518     */ { 00,   02,   20,   0b0101101101010000 }, /*   29   30   29   30   30   29   30   30   29   30   29   30        355
1519     */ { 00,   02,   10,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1520     */ { 08,   01,   30,   0b1001010101110000 }, /*   30   29   29   30   29   30   29   30   29   30   30   30   29   384
1521     */ { 00,   02,   17,   0b1001001011110000 }, /*   30   29   29   30   29   29   30   29   30   30   30   30        355
1522     */ { 00,   02,   07,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1523     */ { 04,   01,   27,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   29   383
1524     */ { 00,   02,   14,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1525     */ { 12,   02,   02,   0b1110101001010000 }, /*   30   30   30   29   30   29   30   29   29   30   29   30   29   384
1526     */ { 00,   02,   21,   0b1101011010010000 }, /*   30   30   29   30   29   30   30   29   30   29   29   30        355
1527     */ { 00,   02,   11,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1528     */ { 10,   02,   01,   0b0010101101101000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29   30   384
1529     */ { 00,   02,   19,   0b0010011011100000 }, /*   29   29   30   29   29   30   30   29   30   30   30   29        354
1530     */ { 00,   02,   08,   0b0101001011100000 }, /*   29   30   29   30   29   29   30   29   30   30   30   29        354
1531     */ { 06,   01,   28,   0b1100100101101000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29   30   384
1532     */ { 00,   02,   16,   0b1100100101010000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30        354
1533     */ { 00,   02,   04,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1534     */ { 02,   01,   24,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30   29   384
1535     */ { 00,   02,   12,   0b1011010110010000 }, /*   30   29   30   30   29   30   29   30   30   29   29   30        355
1536     */ { 12,   02,   02,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30   29   384
1537     */ { 00,   02,   20,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
1538     */ { 00,   02,   10,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1539     */ { 07,   01,   30,   0b1001001011011000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30   30   384
1540     */ { 00,   02,   18,   0b1001001010110000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30        354
1541     */ { 00,   02,   06,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1542     */ { 05,   01,   26,   0b1011010010101000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29   30   384
1543     */ { 00,   02,   14,   0b0110110010100000 }, /*   29   30   30   29   30   30   29   29   30   29   30   29        354
1544     */ { 00,   02,   03,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1545     */ { 01,   01,   23,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30   29   384
1546     */ { 00,   02,   11,   0b0100101110110000 }, /*   29   30   29   29   30   29   30   30   30   29   30   30        355
1547     */ { 09,   02,   01,   0b0010010110111000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30   30   384
1548     */ { 00,   02,   20,   0b0010010101110000 }, /*   29   29   30   29   29   30   29   30   29   30   30   30        354
1549     */ { 00,   02,   08,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1550     */ { 06,   01,   28,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   29   383
1551     */ { 00,   02,   15,   0b1110100101010000 }, /*   30   30   30   29   30   29   29   30   29   30   29   30        355
1552     */ { 00,   02,   05,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1553     */ { 03,   01,   24,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1554     */ { 00,   02,   12,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1555     */ { 11,   02,   02,   0b0100101101101000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29   30   384
1556     */ { 00,   02,   21,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1557     */ { 00,   02,   09,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1558     */ { 07,   01,   30,   0b0101001001101000 }, /*   29   30   29   30   29   29   30   29   29   30   30   29   30   383
1559     */ { 00,   02,   17,   0b1101001001100000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29        354
1560     */ { 00,   02,   06,   0b1101100101010000 }, /*   30   30   29   30   30   29   29   30   29   30   29   30        355
1561     */ { 05,   01,   26,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1562     */ { 00,   02,   14,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1563     */ { 00,   02,   03,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30        355
1564     */ { 02,   01,   24,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1565     */ { 00,   02,   11,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1566     */ { 10,   01,   31,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
1567     */ { 00,   02,   19,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1568     */ { 00,   02,   08,   0b1101001001010000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30        354
1569     */ { 06,   01,   27,   0b1101010101001000 }, /*   30   30   29   30   29   30   29   30   29   30   29   29   30   384
1570     */ { 00,   02,   15,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1571     */ { 00,   02,   05,   0b0011010110100000 }, /*   29   29   30   30   29   30   29   30   30   29   30   29        354
1572     */ { 02,   01,   25,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30   29   384
1573     */ { 00,   02,   12,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1574     */ { 12,   02,   02,   0b0100100110111000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30   30   384
1575     */ { 00,   02,   21,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1576     */ { 00,   02,   10,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1577     */ { 08,   01,   29,   0b1011001001011000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30   30   384
1578     */ { 00,   02,   17,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30        354
1579     */ { 00,   02,   06,   0b0110110101000000 }, /*   29   30   30   29   30   30   29   30   29   30   29   29        354
1580     */ { 04,   01,   26,   0b1010110110101000 }, /*   30   29   30   29   30   30   29   30   30   29   30   29   30   385
1581     */ { 00,   02,   14,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1582     */ { 00,   02,   03,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1583     */ { 02,   01,   24,   0b0100100101111000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30   30   384
1584     */ { 00,   02,   12,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1585     */ { 09,   01,   31,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   29   383
1586     */ { 00,   02,   18,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1587     */ { 00,   02,   07,   0b1110101001010000 }, /*   30   30   30   29   30   29   30   29   29   30   29   30        355
1588     */ { 06,   01,   28,   0b0110101101001000 }, /*   29   30   30   29   30   29   30   30   29   30   29   29   30   384
1589     */ { 00,   02,   15,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1590     */ { 00,   02,   05,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1591     */ { 03,   01,   25,   0b1001001011110000 }, /*   30   29   29   30   29   29   30   29   30   30   30   30   29   384
1592     */ { 00,   02,   13,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1593     */ { 11,   02,   01,   0b1100100101101000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29   30   384
1594     */ { 00,   02,   20,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1595     */ { 00,   02,   09,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1596     */ { 08,   01,   29,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30   29   384
1597     */ { 00,   02,   16,   0b1011010011010000 }, /*   30   29   30   30   29   30   29   29   30   30   29   30        355
1598     */ { 00,   02,   06,   0b0101011010110000 }, /*   29   30   29   30   29   30   30   29   30   29   30   30        355
1599     */ { 04,   01,   27,   0b0010011011011000 }, /*   29   29   30   29   29   30   30   29   30   30   29   30   30   384
1600     */ { 00,   02,   15,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1601     */ { 00,   02,   03,   0b1001001011010000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30        354
1602     */ { 02,   01,   23,   0b1100100110011000 }, /*   30   30   29   29   30   29   29   30   30   29   29   30   30   384
1603     */ { 00,   02,   11,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1604     */ { 09,   01,   31,   0b1011010010101000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29   30   384
1605     */ { 00,   02,   18,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1606     */ { 00,   02,   07,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1607     */ { 06,   01,   28,   0b0101010110101000 }, /*   29   30   29   30   29   30   29   30   30   29   30   29   30   384
1608     */ { 00,   02,   16,   0b0100101110110000 }, /*   29   30   29   29   30   29   30   30   30   29   30   30        355
1609     */ { 00,   02,   05,   0b0010010110110000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30        354
1610     */ { 03,   01,   25,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
1611     */ { 00,   02,   13,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1612     */ { 11,   02,   02,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   29   383
1613     */ { 00,   02,   19,   0b1110100101010000 }, /*   30   30   30   29   30   29   29   30   29   30   29   30        355
1614     */ { 00,   02,   09,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1615     */ { 08,   01,   29,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1616     */ { 00,   02,   17,   0b1001101101010000 }, /*   30   29   29   30   30   29   30   30   29   30   29   30        355
1617     */ { 00,   02,   06,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1618     */ { 04,   01,   26,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
1619     */ { 00,   02,   14,   0b1010010011110000 }, /*   30   29   30   29   29   30   29   29   30   30   30   30        355
1620     */ { 00,   02,   04,   0b0101001001100000 }, /*   29   30   29   30   29   29   30   29   29   30   30   29        353
1621     */ { 02,   01,   22,   0b1110100100110000 }, /*   30   30   30   29   30   29   29   30   29   29   30   30   29   384
1622     */ { 00,   02,   10,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1623     */ { 10,   01,   31,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1624     */ { 00,   02,   19,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1625     */ { 00,   02,   07,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30        355
1626     */ { 06,   01,   28,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1627     */ { 00,   02,   16,   0b0100100111100000 }, /*   29   30   29   29   30   29   29   30   30   30   30   29        354
1628     */ { 00,   02,   05,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1629     */ { 04,   01,   24,   0b1101001001101000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29   30   384
1630     */ { 00,   02,   12,   0b1101001001010000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30        354
1631     */ { 11,   02,   01,   0b1101010100101000 }, /*   30   30   29   30   29   30   29   30   29   29   30   29   30   384
1632     */ { 00,   02,   20,   0b1011010101000000 }, /*   30   29   30   30   29   30   29   30   29   30   29   29        354
1633     */ { 00,   02,   08,   0b1011010110100000 }, /*   30   29   30   30   29   30   29   30   30   29   30   29        355
1634     */ { 08,   01,   29,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30   29   384
1635     */ { 00,   02,   17,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1636     */ { 00,   02,   07,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1637     */ { 04,   01,   26,   0b1010010010111000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30   30   384
1638     */ { 00,   02,   14,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1639     */ { 00,   02,   03,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30        354
1640     */ { 01,   01,   23,   0b1011010100101000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29   30   384
1641     */ { 00,   02,   10,   0b0110110101000000 }, /*   29   30   30   29   30   30   29   30   29   30   29   29        354
1642     */ { 11,   01,   30,   0b1010110110101000 }, /*   30   29   30   29   30   30   29   30   30   29   30   29   30   385
1643     */ { 00,   02,   19,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1644     */ { 00,   02,   08,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1645     */ { 06,   01,   28,   0b0100100101111000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30   30   384
1646     */ { 00,   02,   16,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1647     */ { 00,   02,   05,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1648     */ { 03,   01,   25,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   29   383
1649     */ { 00,   02,   11,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30        355
1650     */ { 11,   02,   01,   0b0110101101001000 }, /*   29   30   30   29   30   29   30   30   29   30   29   29   30   384
1651     */ { 00,   02,   20,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1652     */ { 00,   02,   10,   0b0010101110100000 }, /*   29   29   30   29   30   29   30   30   30   29   30   29        354
1653     */ { 07,   01,   29,   0b1001001011110000 }, /*   30   29   29   30   29   29   30   29   30   30   30   30   29   384
1654     */ { 00,   02,   17,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1655     */ { 00,   02,   06,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1656     */ { 05,   01,   26,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1657     */ { 00,   02,   13,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1658     */ { 00,   02,   02,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30        355
1659     */ { 03,   01,   23,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1660     */ { 00,   02,   11,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1661     */ { 07,   01,   30,   0b1010011011010000 }, /*   30   29   30   29   29   30   30   29   30   30   29   30   29   384
1662     */ { 00,   02,   18,   0b1010010111010000 }, /*   30   29   30   29   29   30   29   30   30   30   29   30        355
1663     */ { 00,   02,   08,   0b1001001011010000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30        354
1664     */ { 06,   01,   28,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1665     */ { 00,   02,   15,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1666     */ { 00,   02,   04,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1667     */ { 04,   01,   24,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1668     */ { 00,   02,   12,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1669     */ { 00,   02,   01,   0b0101010110100000 }, /*   29   30   29   30   29   30   29   30   30   29   30   29        354
1670     */ { 02,   01,   21,   0b1010010111010000 }, /*   30   29   30   29   29   30   29   30   30   30   29   30   29   384
1671     */ { 00,   02,   09,   0b1010010110110000 }, /*   30   29   30   29   29   30   29   30   30   29   30   30        355
1672     */ { 07,   01,   30,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
1673     */ { 00,   02,   17,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1674     */ { 00,   02,   06,   0b0110100100110000 }, /*   29   30   30   29   30   29   29   30   29   29   30   30        354
1675     */ { 05,   01,   26,   0b0111010010101000 }, /*   29   30   30   30   29   30   29   29   30   29   30   29   30   384
1676     */ { 00,   02,   14,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1677     */ { 00,   02,   02,   0b1010110110010000 }, /*   30   29   30   29   30   30   29   30   30   29   29   30        355
1678     */ { 03,   01,   23,   0b0100110110101000 }, /*   29   30   29   29   30   30   29   30   30   29   30   29   30   384
1679     */ { 00,   02,   11,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1680     */ { 08,   01,   31,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
1681     */ { 00,   02,   18,   0b1010010011100000 }, /*   30   29   30   29   29   30   29   29   30   30   30   29        354
1682     */ { 00,   02,   07,   0b1101001001100000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29        354
1683     */ { 06,   01,   27,   0b1110100100110000 }, /*   30   30   30   29   30   29   29   30   29   29   30   30   29   384
1684     */ { 00,   02,   15,   0b1101010100100000 }, /*   30   30   29   30   29   30   29   30   29   29   30   29        354
1685     */ { 00,   02,   03,   0b1101101010100000 }, /*   30   30   29   30   30   29   30   29   30   29   30   29        355
1686     */ { 04,   01,   24,   0b0110101101010000 }, /*   29   30   30   29   30   29   30   30   29   30   29   30   29   384
1687     */ { 00,   02,   12,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1688     */ { 00,   02,   02,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1689     */ { 03,   01,   21,   0b1010010011101000 }, /*   30   29   30   29   29   30   29   29   30   30   30   29   30   384
1690     */ { 00,   02,   09,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1691     */ { 07,   01,   29,   0b1101001001011000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30   30   384
1692     */ { 00,   02,   17,   0b1011001001010000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30        354
1693     */ { 00,   02,   05,   0b1101010100100000 }, /*   30   30   29   30   29   30   29   30   29   29   30   29        354
1694     */ { 05,   01,   25,   0b1101011010100000 }, /*   30   30   29   30   29   30   30   29   30   29   30   29   29   384
1695     */ { 00,   02,   13,   0b1011010110100000 }, /*   30   29   30   30   29   30   29   30   30   29   30   29        355
1696     */ { 00,   02,   03,   0b0101010111010000 }, /*   29   30   29   30   29   30   29   30   30   30   29   30        355
1697     */ { 03,   01,   23,   0b0100101011011000 }, /*   29   30   29   29   30   29   30   29   30   30   29   30   30   384
1698     */ { 00,   02,   11,   0b0100100111010000 }, /*   29   30   29   29   30   29   29   30   30   30   29   30        354
1699     */ { 07,   01,   31,   0b1010010010111000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30   30   384
1700     */ { 00,   02,   19,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1701     */ { 00,   02,   08,   0b1010101001010000 }, /*   30   29   30   29   30   29   30   29   29   30   29   30        354
1702     */ { 06,   01,   28,   0b1011010100101000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29   30   384
1703     */ { 00,   02,   16,   0b0110110100100000 }, /*   29   30   30   29   30   30   29   30   29   29   30   29        354
1704     */ { 00,   02,   05,   0b1010110110100000 }, /*   30   29   30   29   30   30   29   30   30   29   30   29        355
1705     */ { 04,   01,   25,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30   29   384
1706     */ { 00,   02,   13,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1707     */ { 00,   02,   03,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1708     */ { 03,   01,   23,   0b0110010010111000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   30   384
1709     */ { 00,   02,   10,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1710     */ { 07,   01,   30,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   29   383
1711     */ { 00,   02,   17,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30        355
1712     */ { 00,   02,   07,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1713     */ { 05,   01,   26,   0b1010101101100000 }, /*   30   29   30   29   30   29   30   30   29   30   30   29   29   384
1714     */ { 00,   02,   14,   0b1010101011100000 }, /*   30   29   30   29   30   29   30   29   30   30   30   29        355
1715     */ { 00,   02,   04,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1716     */ { 03,   01,   24,   0b1100100101110000 }, /*   30   30   29   29   30   29   29   30   29   30   30   30   29   384
1717     */ { 00,   02,   11,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
1718     */ { 08,   01,   31,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1719     */ { 00,   02,   19,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1720     */ { 00,   02,   08,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1721     */ { 06,   01,   28,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
1722     */ { 00,   02,   16,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1723     */ { 00,   02,   05,   0b1010011011010000 }, /*   30   29   30   29   29   30   30   29   30   30   29   30        355
1724     */ { 04,   01,   26,   0b0101001011101000 }, /*   29   30   29   30   29   29   30   29   30   30   30   29   30   384
1725     */ { 00,   02,   13,   0b0101001011010000 }, /*   29   30   29   30   29   29   30   29   30   30   29   30        354
1726     */ { 00,   02,   02,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1727     */ { 03,   01,   22,   0b1110010010101000 }, /*   30   30   30   29   29   30   29   29   30   29   30   29   30   384
1728     */ { 00,   02,   10,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1729     */ { 07,   01,   29,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1730     */ { 00,   02,   17,   0b1001110101010000 }, /*   30   29   29   30   30   30   29   30   29   30   29   30        355
1731     */ { 00,   02,   07,   0b0101010110100000 }, /*   29   30   29   30   29   30   29   30   30   29   30   29        354
1732     */ { 05,   01,   27,   0b1010010111010000 }, /*   30   29   30   29   29   30   29   30   30   30   29   30   29   384
1733     */ { 00,   02,   14,   0b1010010110110000 }, /*   30   29   30   29   29   30   29   30   30   29   30   30        355
1734     */ { 00,   02,   04,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1735     */ { 04,   01,   24,   0b1010100010111000 }, /*   30   29   30   29   30   29   29   29   30   29   30   30   30   384
1736     */ { 00,   02,   12,   0b0110100100110000 }, /*   29   30   30   29   30   29   29   30   29   29   30   30        354
1737     */ { 09,   01,   31,   0b0110101010011000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30   30   384
1738     */ { 00,   02,   19,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1739     */ { 00,   02,   08,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1740     */ { 06,   01,   29,   0b0100110110101000 }, /*   29   30   29   29   30   30   29   30   30   29   30   29   30   384
1741     */ { 00,   02,   16,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1742     */ { 00,   02,   05,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1743     */ { 04,   01,   26,   0b0101000101110000 }, /*   29   30   29   30   29   29   29   30   29   30   30   30   29   383
1744     */ { 00,   02,   13,   0b1101000101100000 }, /*   30   30   29   30   29   29   29   30   29   30   30   29        354
1745     */ { 00,   02,   01,   0b1110100100110000 }, /*   30   30   30   29   30   29   29   30   29   29   30   30        355
1746     */ { 03,   01,   22,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30   29   383
1747     */ { 00,   02,   09,   0b1101101010100000 }, /*   30   30   29   30   30   29   30   29   30   29   30   29        355
1748     */ { 07,   01,   30,   0b0101101101010000 }, /*   29   30   29   30   30   29   30   30   29   30   29   30   29   384
1749     */ { 00,   02,   17,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1750     */ { 00,   02,   07,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1751     */ { 05,   01,   27,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
1752     */ { 00,   02,   15,   0b1010001011010000 }, /*   30   29   30   29   29   29   30   29   30   30   29   30        354
1753     */ { 00,   02,   03,   0b1101000101010000 }, /*   30   30   29   30   29   29   29   30   29   30   29   30        354
1754     */ { 04,   01,   23,   0b1101010100101000 }, /*   30   30   29   30   29   30   29   30   29   29   30   29   30   384
1755     */ { 00,   02,   11,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1756     */ { 09,   01,   31,   0b1101011010010000 }, /*   30   30   29   30   29   30   30   29   30   29   29   30   29   384
1757     */ { 00,   02,   18,   0b1010110110100000 }, /*   30   29   30   29   30   30   29   30   30   29   30   29        355
1758     */ { 00,   02,   08,   0b0101010111010000 }, /*   29   30   29   30   29   30   29   30   30   30   29   30        355
1759     */ { 06,   01,   29,   0b0010101011011000 }, /*   29   29   30   29   30   29   30   29   30   30   29   30   30   384
1760     */ { 00,   02,   17,   0b0100010110110000 }, /*   29   30   29   29   29   30   29   30   30   29   30   30        354
1761     */ { 00,   02,   05,   0b1010001010110000 }, /*   30   29   30   29   29   29   30   29   30   29   30   30        354
1762     */ { 05,   01,   25,   0b1011000101011000 }, /*   30   29   30   30   29   29   29   30   29   30   29   30   30   384
1763     */ { 00,   02,   13,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1764     */ { 00,   02,   02,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1765     */ { 02,   01,   21,   0b1011010110010000 }, /*   30   29   30   30   29   30   29   30   30   29   29   30   29   384
1766     */ { 00,   02,   09,   0b1010110101100000 }, /*   30   29   30   29   30   30   29   30   29   30   30   29        355
1767     */ { 07,   01,   30,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30   29   384
1768     */ { 00,   02,   18,   0b0101001101110000 }, /*   29   30   29   30   29   29   30   30   29   30   30   30        355
1769     */ { 00,   02,   07,   0b0100010101110000 }, /*   29   30   29   29   29   30   29   30   29   30   30   30        354
1770     */ { 05,   01,   27,   0b0110001010111000 }, /*   29   30   30   29   29   29   30   29   30   29   30   30   30   384
1771     */ { 00,   02,   15,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1772     */ { 00,   02,   04,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1773     */ { 03,   01,   23,   0b0110110010101000 }, /*   29   30   30   29   30   30   29   29   30   29   30   29   30   384
1774     */ { 00,   02,   11,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1775     */ { 10,   01,   31,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30   29   384
1776     */ { 00,   02,   19,   0b1010011011010000 }, /*   30   29   30   29   29   30   30   29   30   30   29   30        355
1777     */ { 00,   02,   08,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1778     */ { 06,   01,   28,   0b1001010101110000 }, /*   30   29   29   30   29   30   29   30   29   30   30   30   29   384
1779     */ { 00,   02,   16,   0b1010010101100000 }, /*   30   29   30   29   29   30   29   30   29   30   30   29        354
1780     */ { 00,   02,   05,   0b1101001010100000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29        354
1781     */ { 05,   01,   24,   0b1110100101010000 }, /*   30   30   30   29   30   29   29   30   29   30   29   30   29   384
1782     */ { 00,   02,   12,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1783     */ { 00,   02,   02,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1784     */ { 03,   01,   22,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30   29   384
1785     */ { 00,   02,   09,   0b1010011011010000 }, /*   30   29   30   29   29   30   30   29   30   30   29   30        355
1786     */ { 07,   01,   30,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1787     */ { 00,   02,   18,   0b0100101010110000 }, /*   29   30   29   29   30   29   30   29   30   29   30   30        354
1788     */ { 00,   02,   07,   0b1010100011010000 }, /*   30   29   30   29   30   29   29   29   30   30   29   30        354
1789     */ { 05,   01,   26,   0b1101001010101000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29   30   384
1790     */ { 00,   02,   14,   0b1011001010100000 }, /*   30   29   30   30   29   29   30   29   30   29   30   29        354
1791     */ { 00,   02,   03,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1792     */ { 04,   01,   24,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1793     */ { 00,   02,   11,   0b0100110110100000 }, /*   29   30   29   29   30   30   29   30   30   29   30   29        354
1794     */ { 00,   01,   31,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1795     */ { 02,   01,   21,   0b0100101010111000 }, /*   29   30   29   29   30   29   30   29   30   29   30   30   30   384
1796     */ { 00,   02,   09,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1797     */ { 06,   01,   28,   0b1010100010111000 }, /*   30   29   30   29   30   29   29   29   30   29   30   30   30   384
1798     */ { 00,   02,   16,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1799     */ { 00,   02,   05,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30        354
1800     */ { 04,   01,   25,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1801     */ { 00,   02,   13,   0b0110101101010000 }, /*   29   30   30   29   30   29   30   30   29   30   29   30        355
1802     */ { 00,   02,   03,   0b0010101110100000 }, /*   29   29   30   29   30   29   30   30   30   29   30   29        354
1803     */ { 02,   01,   23,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30   29   384
1804     */ { 00,   02,   11,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1805     */ { 06,   01,   31,   0b0101000101110000 }, /*   29   30   29   30   29   29   29   30   29   30   30   30   29   383
1806     */ { 00,   02,   18,   0b1101000101100000 }, /*   30   30   29   30   29   29   29   30   29   30   30   29        354
1807     */ { 00,   02,   07,   0b1110010010110000 }, /*   30   30   30   29   29   30   29   29   30   29   30   30        355
1808     */ { 05,   01,   28,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30   29   383
1809     */ { 00,   02,   14,   0b1101101010010000 }, /*   30   30   29   30   30   29   30   29   30   29   29   30        355
1810     */ { 00,   02,   04,   0b0101101101010000 }, /*   29   30   29   30   30   29   30   30   29   30   29   30        355
1811     */ { 03,   01,   25,   0b0010101101101000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29   30   384
1812     */ { 00,   02,   13,   0b0010101011100000 }, /*   29   29   30   29   30   29   30   29   30   30   30   29        354
1813     */ { 00,   02,   01,   0b1010001011100000 }, /*   30   29   30   29   29   29   30   29   30   30   30   29        354
1814     */ { 02,   01,   21,   0b1101000101101000 }, /*   30   30   29   30   29   29   29   30   29   30   30   29   30   384
1815     */ { 00,   02,   09,   0b1100100101010000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30        354
1816     */ { 06,   01,   29,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1817     */ { 00,   02,   16,   0b1011010100100000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29        354
1818     */ { 00,   02,   05,   0b1011011010010000 }, /*   30   29   30   30   29   30   30   29   30   29   29   30        355
1819     */ { 04,   01,   26,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30   29   384
1820     */ { 00,   02,   14,   0b0101010111010000 }, /*   29   30   29   30   29   30   29   30   30   30   29   30        355
1821     */ { 00,   02,   03,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1822     */ { 03,   01,   23,   0b1010001011011000 }, /*   30   29   30   29   29   29   30   29   30   30   29   30   30   384
1823     */ { 00,   02,   11,   0b1010001010110000 }, /*   30   29   30   29   29   29   30   29   30   29   30   30        354
1824     */ { 07,   01,   31,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1825     */ { 00,   02,   18,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1826     */ { 00,   02,   07,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1827     */ { 05,   01,   27,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30   29   384
1828     */ { 00,   02,   15,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
1829     */ { 00,   02,   04,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
1830     */ { 04,   01,   25,   0b0010010110111000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30   30   384
1831     */ { 00,   02,   13,   0b0010010101110000 }, /*   29   29   30   29   29   30   29   30   29   30   30   30        354
1832     */ { 09,   02,   02,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
1833     */ { 00,   02,   20,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1834     */ { 00,   02,   09,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1835     */ { 06,   01,   29,   0b0110110010101000 }, /*   29   30   30   29   30   30   29   29   30   29   30   29   30   384
1836     */ { 00,   02,   17,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1837     */ { 00,   02,   05,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1838     */ { 04,   01,   26,   0b0101001101101000 }, /*   29   30   29   30   29   29   30   30   29   30   30   29   30   384
1839     */ { 00,   02,   14,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1840     */ { 00,   02,   03,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1841     */ { 03,   01,   23,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   29   383
1842     */ { 00,   02,   10,   0b1101001010100000 }, /*   30   30   29   30   29   29   30   29   30   29   30   29        354
1843     */ { 07,   01,   30,   0b1110100101010000 }, /*   30   30   30   29   30   29   29   30   29   30   29   30   29   384
1844     */ { 00,   02,   18,   0b1101010101010000 }, /*   30   30   29   30   29   30   29   30   29   30   29   30        355
1845     */ { 00,   02,   07,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
1846     */ { 05,   01,   27,   0b1010101011010000 }, /*   30   29   30   29   30   29   30   29   30   30   29   30   29   384
1847     */ { 00,   02,   15,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1848     */ { 00,   02,   05,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1849     */ { 04,   01,   24,   0b1010010101011000 }, /*   30   29   30   29   29   30   29   30   29   30   29   30   30   384
1850     */ { 00,   02,   12,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1851     */ { 08,   02,   01,   0b1101001001011000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30   30   384
1852     */ { 00,   02,   20,   0b1011001010010000 }, /*   30   29   30   30   29   29   30   29   30   29   29   30        354
1853     */ { 00,   02,   08,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1854     */ { 07,   01,   29,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1855     */ { 00,   02,   17,   0b0010110110100000 }, /*   29   29   30   29   30   30   29   30   30   29   30   29        354
1856     */ { 00,   02,   06,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1857     */ { 05,   01,   26,   0b0100101010111000 }, /*   29   30   29   29   30   29   30   29   30   29   30   30   30   384
1858     */ { 00,   02,   14,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1859     */ { 00,   02,   03,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1860     */ { 03,   01,   23,   0b1011001001011000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30   30   384
1861     */ { 00,   02,   10,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30        354
1862     */ { 08,   01,   30,   0b1010110101001000 }, /*   30   29   30   29   30   30   29   30   29   30   29   29   30   384
1863     */ { 00,   02,   18,   0b0110101101010000 }, /*   29   30   30   29   30   29   30   30   29   30   29   30        355
1864     */ { 00,   02,   08,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1865     */ { 05,   01,   27,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30   29   384
1866     */ { 00,   02,   15,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30        355
1867     */ { 00,   02,   05,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1868     */ { 04,   01,   25,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   29   383
1869     */ { 00,   02,   11,   0b1110010010100000 }, /*   30   30   30   29   29   30   29   29   30   29   30   29        354
1870     */ { 10,   01,   31,   0b1110101001010000 }, /*   30   30   30   29   30   29   30   29   29   30   29   30   29   384
1871     */ { 00,   02,   19,   0b1101101010010000 }, /*   30   30   29   30   30   29   30   29   30   29   29   30        355
1872     */ { 00,   02,   09,   0b0101101011010000 }, /*   29   30   29   30   30   29   30   29   30   30   29   30        355
1873     */ { 06,   01,   29,   0b0010101101101000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29   30   384
1874     */ { 00,   02,   17,   0b0010101011100000 }, /*   29   29   30   29   30   29   30   29   30   30   30   29        354
1875     */ { 00,   02,   06,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1876     */ { 05,   01,   26,   0b1100100101101000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29   30   384
1877     */ { 00,   02,   13,   0b1100100101010000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30        354
1878     */ { 00,   02,   02,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1879     */ { 03,   01,   22,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30   29   384
1880     */ { 00,   02,   10,   0b1011011010010000 }, /*   30   29   30   30   29   30   30   29   30   29   29   30        355
1881     */ { 07,   01,   30,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30   29   384
1882     */ { 00,   02,   18,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
1883     */ { 00,   02,   08,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1884     */ { 05,   01,   28,   0b1001001011011000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30   30   384
1885     */ { 00,   02,   15,   0b1001001010110000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30        354
1886     */ { 00,   02,   04,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1887     */ { 04,   01,   24,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1888     */ { 00,   02,   12,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1889     */ { 00,   01,   31,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1890     */ { 02,   01,   21,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1891     */ { 00,   02,   09,   0b0101010110110000 }, /*   29   30   29   30   29   30   29   30   30   29   30   30        355
1892     */ { 06,   01,   30,   0b0010010110111000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30   30   384
1893     */ { 00,   02,   17,   0b0010010101110000 }, /*   29   29   30   29   29   30   29   30   29   30   30   30        354
1894     */ { 00,   02,   06,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1895     */ { 05,   01,   26,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   29   383
1896     */ { 00,   02,   13,   0b1110100101010000 }, /*   30   30   30   29   30   29   29   30   29   30   29   30        355
1897     */ { 00,   02,   02,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1898     */ { 03,   01,   22,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1899     */ { 00,   02,   10,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1900     */ { 08,   01,   31,   0b0100101101101000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29   30   384
1901     */ { 00,   02,   19,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1902     */ { 00,   02,   08,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1903     */ { 05,   01,   29,   0b0101001001101000 }, /*   29   30   29   30   29   29   30   29   29   30   30   29   30   383
1904     */ { 00,   02,   16,   0b1101001001100000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29        354
1905     */ { 00,   02,   04,   0b1101100101010000 }, /*   30   30   29   30   30   29   29   30   29   30   29   30        355
1906     */ { 04,   01,   25,   0b0110101010101000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29   30   384
1907     */ { 00,   02,   13,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1908     */ { 00,   02,   02,   0b1001101011010000 }, /*   30   29   29   30   30   29   30   29   30   30   29   30        355
1909     */ { 02,   01,   22,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1910     */ { 00,   02,   10,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1911     */ { 06,   01,   30,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
1912     */ { 00,   02,   18,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
1913     */ { 00,   02,   06,   0b1101001001010000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30        354
1914     */ { 05,   01,   26,   0b1101100101001000 }, /*   30   30   29   30   30   29   29   30   29   30   29   29   30   384
1915     */ { 00,   02,   14,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1916     */ { 00,   02,   04,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1917     */ { 02,   01,   23,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30   29   384
1918     */ { 00,   02,   11,   0b1001010111010000 }, /*   30   29   29   30   29   30   29   30   30   30   29   30        355
1919     */ { 07,   02,   01,   0b0100101011011000 }, /*   29   30   29   29   30   29   30   29   30   30   29   30   30   384
1920     */ { 00,   02,   20,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1921     */ { 00,   02,   08,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1922     */ { 05,   01,   28,   0b1011001001011000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30   30   384
1923     */ { 00,   02,   16,   0b0110101010010000 }, /*   29   30   30   29   30   29   30   29   30   29   29   30        354
1924     */ { 00,   02,   05,   0b1010110101000000 }, /*   30   29   30   29   30   30   29   30   29   30   29   29        354
1925     */ { 04,   01,   24,   0b1011010110101000 }, /*   30   29   30   30   29   30   29   30   30   29   30   29   30   385
1926     */ { 00,   02,   13,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1927     */ { 00,   02,   02,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1928     */ { 02,   01,   23,   0b0100100110111000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30   30   384
1929     */ { 00,   02,   10,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1930     */ { 06,   01,   30,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30   29   383
1931     */ { 00,   02,   17,   0b1110010010100000 }, /*   30   30   30   29   29   30   29   29   30   29   30   29        354
1932     */ { 00,   02,   06,   0b1110101001010000 }, /*   30   30   30   29   30   29   30   29   29   30   29   30        355
1933     */ { 05,   01,   26,   0b0110110101001000 }, /*   29   30   30   29   30   30   29   30   29   30   29   29   30   384
1934     */ { 00,   02,   14,   0b0101101101010000 }, /*   29   30   29   30   30   29   30   30   29   30   29   30        355
1935     */ { 00,   02,   04,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1936     */ { 03,   01,   24,   0b1001010101110000 }, /*   30   29   29   30   29   30   29   30   29   30   30   30   29   384
1937     */ { 00,   02,   11,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
1938     */ { 07,   01,   31,   0b1100100101101000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29   30   384
1939     */ { 00,   02,   19,   0b1100100101010000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30        354
1940     */ { 00,   02,   08,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
1941     */ { 06,   01,   27,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30   29   384
1942     */ { 00,   02,   15,   0b1011011010010000 }, /*   30   29   30   30   29   30   30   29   30   29   29   30        355
1943     */ { 00,   02,   05,   0b0101011011010000 }, /*   29   30   29   30   29   30   30   29   30   30   29   30        355
1944     */ { 04,   01,   26,   0b0010101011011000 }, /*   29   29   30   29   30   29   30   29   30   30   29   30   30   384
1945     */ { 00,   02,   13,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
1946     */ { 00,   02,   02,   0b1001001011010000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30        354
1947     */ { 02,   01,   22,   0b1100100101011000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30   30   384
1948     */ { 00,   02,   10,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
1949     */ { 07,   01,   29,   0b1101010010101000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29   30   384
1950     */ { 00,   02,   17,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
1951     */ { 00,   02,   06,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
1952     */ { 05,   01,   27,   0b0101011010101000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29   30   384
1953     */ { 00,   02,   14,   0b0100110110110000 }, /*   29   30   29   29   30   30   29   30   30   29   30   30        355
1954     */ { 00,   02,   04,   0b0010010110110000 }, /*   29   29   30   29   29   30   29   30   30   29   30   30        354
1955     */ { 03,   01,   24,   0b1001001010111000 }, /*   30   29   29   30   29   29   30   29   30   29   30   30   30   384
1956     */ { 00,   02,   12,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
1957     */ { 08,   01,   31,   0b1010100101011000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30   30   384
1958     */ { 00,   02,   19,   0b0110100101010000 }, /*   29   30   30   29   30   29   29   30   29   30   29   30        354
1959     */ { 00,   02,   08,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
1960     */ { 06,   01,   28,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30   29   384
1961     */ { 00,   02,   15,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30        355
1962     */ { 00,   02,   05,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
1963     */ { 04,   01,   25,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
1964     */ { 00,   02,   13,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
1965     */ { 00,   02,   02,   0b0101001001110000 }, /*   29   30   29   30   29   29   30   29   29   30   30   30        354
1966     */ { 03,   01,   22,   0b0110100100110000 }, /*   29   30   30   29   30   29   29   30   29   29   30   30   29   383
1967     */ { 00,   02,   09,   0b1101100101010000 }, /*   30   30   29   30   30   29   29   30   29   30   29   30        355
1968     */ { 07,   01,   30,   0b0110101010101000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29   30   384
1969     */ { 00,   02,   17,   0b0101011010100000 }, /*   29   30   29   30   29   30   30   29   30   29   30   29        354
1970     */ { 00,   02,   06,   0b1001101011010000 }, /*   30   29   29   30   30   29   30   29   30   30   29   30        355
1971     */ { 05,   01,   27,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
1972     */ { 00,   02,   15,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
1973     */ { 00,   02,   03,   0b1010010011100000 }, /*   30   29   30   29   29   30   29   29   30   30   30   29        354
1974     */ { 04,   01,   23,   0b1101001001101000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29   30   384
1975     */ { 00,   02,   11,   0b1101001001010000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30        354
1976     */ { 08,   01,   31,   0b1101010101001000 }, /*   30   30   29   30   29   30   29   30   29   30   29   29   30   384
1977     */ { 00,   02,   18,   0b1011010101000000 }, /*   30   29   30   30   29   30   29   30   29   30   29   29        354
1978     */ { 00,   02,   07,   0b1101011010100000 }, /*   30   30   29   30   29   30   30   29   30   29   30   29        355
1979     */ { 06,   01,   28,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30   29   384
1980     */ { 00,   02,   16,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1981     */ { 00,   02,   05,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
1982     */ { 04,   01,   25,   0b1010010011011000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30   30   384
1983     */ { 00,   02,   13,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
1984     */ { 10,   02,   02,   0b1011001001011000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30   30   384
1985     */ { 00,   02,   20,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30        354
1986     */ { 00,   02,   09,   0b0110110101000000 }, /*   29   30   30   29   30   30   29   30   29   30   29   29        354
1987     */ { 06,   01,   29,   0b1011010110101000 }, /*   30   29   30   30   29   30   29   30   30   29   30   29   30   385
1988     */ { 00,   02,   18,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1989     */ { 00,   02,   06,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30        355
1990     */ { 05,   01,   27,   0b0100100110111000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30   30   384
1991     */ { 00,   02,   15,   0b0100100101110000 }, /*   29   30   29   29   30   29   29   30   29   30   30   30        354
1992     */ { 00,   02,   04,   0b0110010010110000 }, /*   29   30   30   29   29   30   29   29   30   29   30   30        354
1993     */ { 03,   01,   23,   0b0110101001010000 }, /*   29   30   30   29   30   29   30   29   29   30   29   30   29   383
1994     */ { 00,   02,   10,   0b1110101001010000 }, /*   30   30   30   29   30   29   30   29   29   30   29   30        355
1995     */ { 08,   01,   31,   0b0110110101001000 }, /*   29   30   30   29   30   30   29   30   29   30   29   29   30   384
1996     */ { 00,   02,   19,   0b0101101011010000 }, /*   29   30   29   30   30   29   30   29   30   30   29   30        355
1997     */ { 00,   02,   08,   0b0010101101100000 }, /*   29   29   30   29   30   29   30   30   29   30   30   29        354
1998     */ { 05,   01,   28,   0b1001001101110000 }, /*   30   29   29   30   29   29   30   30   29   30   30   30   29   384
1999     */ { 00,   02,   16,   0b1001001011100000 }, /*   30   29   29   30   29   29   30   29   30   30   30   29        354
2000     */ { 00,   02,   05,   0b1100100101100000 }, /*   30   30   29   29   30   29   29   30   29   30   30   29        354
2001     */ { 04,   01,   24,   0b1110010010101000 }, /*   30   30   30   29   29   30   29   29   30   29   30   29   30   384
2002     */ { 00,   02,   12,   0b1101010010100000 }, /*   30   30   29   30   29   30   29   29   30   29   30   29        354
2003     */ { 00,   02,   01,   0b1101101001010000 }, /*   30   30   29   30   30   29   30   29   29   30   29   30        355
2004     */ { 02,   01,   22,   0b0101101010101000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29   30   384
2005     */ { 00,   02,   09,   0b0101011011000000 }, /*   29   30   29   30   29   30   30   29   30   30   29   29        354
2006     */ { 07,   01,   29,   0b1010101011011000 }, /*   30   29   30   29   30   29   30   29   30   30   29   30   30   385
2007     */ { 00,   02,   18,   0b0010010111010000 }, /*   29   29   30   29   29   30   29   30   30   30   29   30        354
2008     */ { 00,   02,   07,   0b1001001011010000 }, /*   30   29   29   30   29   29   30   29   30   30   29   30        354
2009     */ { 05,   01,   26,   0b1100100101011000 }, /*   30   30   29   29   30   29   29   30   29   30   29   30   30   384
2010     */ { 00,   02,   14,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
2011     */ { 00,   02,   03,   0b1011010010100000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29        354
2012     */ { 03,   01,   23,   0b1011101001010000 }, /*   30   29   30   30   30   29   30   29   29   30   29   30   29   384
2013     */ { 00,   02,   10,   0b1011010101010000 }, /*   30   29   30   30   29   30   29   30   29   30   29   30        355
2014     */ { 09,   01,   31,   0b0101010110101000 }, /*   29   30   29   30   29   30   29   30   30   29   30   29   30   384
2015     */ { 00,   02,   19,   0b0100101110100000 }, /*   29   30   29   29   30   29   30   30   30   29   30   29        354
2016     */ { 00,   02,   08,   0b1010010110110000 }, /*   30   29   30   29   29   30   29   30   30   29   30   30        355
2017     */ { 05,   01,   28,   0b0101001010111000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30   30   384
2018     */ { 00,   02,   16,   0b0101001010110000 }, /*   29   30   29   30   29   29   30   29   30   29   30   30        354
2019     */ { 00,   02,   05,   0b1010100101010000 }, /*   30   29   30   29   30   29   29   30   29   30   29   30        354
2020     */ { 04,   01,   25,   0b1011010010101000 }, /*   30   29   30   30   29   30   29   29   30   29   30   29   30   384
2021     */ { 00,   02,   12,   0b0110101010100000 }, /*   29   30   30   29   30   29   30   29   30   29   30   29        354
2022     */ { 00,   02,   01,   0b1010110101010000 }, /*   30   29   30   29   30   30   29   30   29   30   29   30        355
2023     */ { 02,   01,   22,   0b0101010110101000 }, /*   29   30   29   30   29   30   29   30   30   29   30   29   30   384
2024     */ { 00,   02,   10,   0b0100101101100000 }, /*   29   30   29   29   30   29   30   30   29   30   30   29        354
2025     */ { 06,   01,   29,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30   29   384
2026     */ { 00,   02,   17,   0b1010010101110000 }, /*   30   29   30   29   29   30   29   30   29   30   30   30        355
2027     */ { 00,   02,   07,   0b0101001001110000 }, /*   29   30   29   30   29   29   30   29   29   30   30   30        354
2028     */ { 05,   01,   27,   0b0110100100110000 }, /*   29   30   30   29   30   29   29   30   29   29   30   30   29   383
2029     */ { 00,   02,   13,   0b1101100100110000 }, /*   30   30   29   30   30   29   29   30   29   29   30   30        355
2030     */ { 00,   02,   03,   0b0101101010100000 }, /*   29   30   29   30   30   29   30   29   30   29   30   29        354
2031     */ { 03,   01,   23,   0b1010101101010000 }, /*   30   29   30   29   30   29   30   30   29   30   29   30   29   384
2032     */ { 00,   02,   11,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30        355
2033     */ { 11,   01,   31,   0b0100101011101000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29   30   384
2034     */ { 00,   02,   19,   0b0100101011100000 }, /*   29   30   29   29   30   29   30   29   30   30   30   29        354
2035     */ { 00,   02,   08,   0b1010010011010000 }, /*   30   29   30   29   29   30   29   29   30   30   29   30        354
2036     */ { 06,   01,   28,   0b1101001001101000 }, /*   30   30   29   30   29   29   30   29   29   30   30   29   30   384
2037     */ { 00,   02,   15,   0b1101001001010000 }, /*   30   30   29   30   29   29   30   29   29   30   29   30        354
2038     */ { 00,   02,   04,   0b1101010100100000 }, /*   30   30   29   30   29   30   29   30   29   29   30   29        354
2039     */ { 05,   01,   24,   0b1101101010100000 }, /*   30   30   29   30   30   29   30   29   30   29   30   29   29   384
2040     */ { 00,   02,   12,   0b1011011010100000 }, /*   30   29   30   30   29   30   30   29   30   29   30   29        355
2041     */ { 00,   02,   01,   0b1001011011010000 }, /*   30   29   29   30   29   30   30   29   30   30   29   30        355
2042     */ { 02,   01,   22,   0b0100101011011000 }, /*   29   30   29   29   30   29   30   29   30   30   29   30   30   384
2043     */ { 00,   02,   10,   0b0100100110110000 }, /*   29   30   29   29   30   29   29   30   30   29   30   30        354
2044     */ { 07,   01,   30,   0b1010010010111000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30   30   384
2045     */ { 00,   02,   17,   0b1010010010110000 }, /*   30   29   30   29   29   30   29   29   30   29   30   30        354
2046     */ { 00,   02,   06,   0b1011001001010000 }, /*   30   29   30   30   29   29   30   29   29   30   29   30        354
2047     */ { 05,   01,   26,   0b1011010100101000 }, /*   30   29   30   30   29   30   29   30   29   29   30   29   30   384
2048     */ { 00,   02,   14,   0b0110110101000000 }, /*   29   30   30   29   30   30   29   30   29   30   29   29        354
2049     */ { 00,   02,   02,   0b1010110110100000 }, /*   30   29   30   29   30   30   29   30   30   29   30   29        355
2050     */ { 03,   01,   23,   0b1001010110110000 }, /*   30   29   29   30   29   30   29   30   30   29   30   30   29   384
         */ };

        internal override int MinCalendarYear => MinLunisolarYear;

        internal override int MaxCalendarYear => MaxLunisolarYear;

        internal override DateTime MinDate => s_minDate;

        internal override DateTime MaxDate => s_maxDate;

        internal override EraInfo[]? CalEraInfo => null;

        internal override int GetYearInfo(int lunarYear, int index)
        {
            if (lunarYear < MinLunisolarYear || lunarYear > MaxLunisolarYear)
            {
                throw new ArgumentOutOfRangeException(
                    "year",
                    lunarYear,
                    SR.Format(SR.ArgumentOutOfRange_Range, MinLunisolarYear, MaxLunisolarYear));
            }

            return s_yinfo[lunarYear - MinLunisolarYear, index];
        }

        internal override int GetYear(int year, DateTime time)
        {
            return year;
        }

        internal override int GetGregorianYear(int year, int era)
        {
            if (era != CurrentEra && era != GregorianEra)
            {
                throw new ArgumentOutOfRangeException(nameof(era), era, SR.ArgumentOutOfRange_InvalidEraValue);
            }
            if (year < MinLunisolarYear || year > MaxLunisolarYear)
            {
                throw new ArgumentOutOfRangeException(
                    nameof(year),
                    year,
                    SR.Format(SR.ArgumentOutOfRange_Range, MinLunisolarYear, MaxLunisolarYear));
            }

            return year;
        }

        public KoreanLunisolarCalendar()
        {
        }

        public override int GetEra(DateTime time)
        {
            CheckTicksRange(time.Ticks);
            return GregorianEra;
        }

        internal override CalendarId BaseCalendarID => CalendarId.KOREA;

        internal override CalendarId ID => CalendarId.KOREANLUNISOLAR;

        public override int[] Eras => [GregorianEra];
    }
}
