// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Xunit;

namespace System.Security.Cryptography.Tests
{
    public class KmacXof256Tests : KmacTestDriver<KmacXof256Tests.Traits, KmacXof256>
    {
        public class Traits : IKmacTrait<KmacXof256>
        {
            public static KmacXof256 Create(byte[] key, byte[] customizationString) =>
                new KmacXof256(key, customizationString);

            public static KmacXof256 Create(ReadOnlySpan<byte> key, ReadOnlySpan<byte> customizationString) =>
                new KmacXof256(key, customizationString);

            public static bool IsSupported => KmacXof256.IsSupported;

            public static void AppendData(KmacXof256 kmac, byte[] data) => kmac.AppendData(data);
            public static void AppendData(KmacXof256 kmac, ReadOnlySpan<byte> data) => kmac.AppendData(data);
            public static byte[] GetHashAndReset(KmacXof256 kmac, int outputLength) => kmac.GetHashAndReset(outputLength);
            public static void GetHashAndReset(KmacXof256 kmac, Span<byte> destination) => kmac.GetHashAndReset(destination);
            public static byte[] GetCurrentHash(KmacXof256 kmac, int outputLength) => kmac.GetCurrentHash(outputLength);
            public static void GetCurrentHash(KmacXof256 kmac, Span<byte> destination) => kmac.GetCurrentHash(destination);
            public static KmacXof256 Clone(KmacXof256 kmac) => kmac.Clone();

            public static byte[] HashData(byte[] key, byte[] source, int outputLength, byte[] customizationString) =>
                KmacXof256.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                KmacXof256.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, ReadOnlySpan<byte> source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                KmacXof256.HashData(key, source, destination, customizationString);

            public static byte[] HashData(byte[] key, Stream source, int outputLength, byte[] customizationString) =>
                KmacXof256.HashData(key, source, outputLength, customizationString);

            public static byte[] HashData(ReadOnlySpan<byte> key, Stream source, int outputLength, ReadOnlySpan<byte> customizationString) =>
                KmacXof256.HashData(key, source, outputLength, customizationString);

            public static void HashData(ReadOnlySpan<byte> key, Stream source, Span<byte> destination, ReadOnlySpan<byte> customizationString) =>
                KmacXof256.HashData(key, source, destination, customizationString);

            public static ValueTask HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                Memory<byte> destination,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return KmacXof256.HashDataAsync(key, source, destination, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                ReadOnlyMemory<byte> key,
                Stream source,
                int outputLength,
                ReadOnlyMemory<byte> customizationString,
                CancellationToken cancellationToken)
            {
                return KmacXof256.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }

            public static ValueTask<byte[]> HashDataAsync(
                byte[] key,
                Stream source,
                int outputLength,
                byte[] customizationString,
                CancellationToken cancellationToken)
            {
                return KmacXof256.HashDataAsync(key, source, outputLength, customizationString, cancellationToken);
            }
        }

        protected override IEnumerable<KmacTestVector> TestVectors
        {
            get
            {
                // Generated test vectors.
                // See KmacTestDriver for how they were generated.
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d8");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e5");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "55");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f453");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca2");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7ac");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3a");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd102");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb7");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf2");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f44");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db870");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15b");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf9");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25db");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf41335");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f1");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87d");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160a");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b25380");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf2");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa65");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502d");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b8");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec3982");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42e");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d38");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec93");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990a");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330a");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e392404");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df95985448");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4f");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463cd8f6dca118f8ee9d3f312fd383d1b9d9");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648513779a7851eeeda44f8032dab43503d");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02a4ad50e5f9a1fd90faaedc1650194a0c");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4ff4ed344fa5a7c0014b3b23b026cecc91");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5240bc461232dce441bacb88c4244fe65");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc6ea56fcdad6adafeab3444bbc005dcd0");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463cd8f6dca118f8ee9d3f312fd383d1b9d9be3f8269da558f4932e91c20ab09870c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648513779a7851eeeda44f8032dab43503d0cd9ef5f7503078012f1486c5bf7704d");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02a4ad50e5f9a1fd90faaedc1650194a0c21319b5d755a31a20d592fba4237ec39");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4ff4ed344fa5a7c0014b3b23b026cecc91e1a215aa34187b33b6c2c62fac21f999");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5240bc461232dce441bacb88c4244fe650b39e93e7b2e2e3111f9035114cf7f03");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc6ea56fcdad6adafeab3444bbc005dcd0cf997742cf81e468961a3f6611cbf7b8");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463cd8f6dca118f8ee9d3f312fd383d1b9d9be3f8269da558f4932e91c20ab09870c832501ff5fb5a6c2ef86724db449b7a1");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648513779a7851eeeda44f8032dab43503d0cd9ef5f7503078012f1486c5bf7704d4232ef92ec3c6e079a4729b4cc202ce8");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02a4ad50e5f9a1fd90faaedc1650194a0c21319b5d755a31a20d592fba4237ec39fab4b67b3a050379d4b4fb6311fa0733");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4ff4ed344fa5a7c0014b3b23b026cecc91e1a215aa34187b33b6c2c62fac21f9997ed176f34460e4c853d7ab401a3b9ba4");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5240bc461232dce441bacb88c4244fe650b39e93e7b2e2e3111f9035114cf7f03912a67e4b6467152ce3704212ec0c0ef");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc6ea56fcdad6adafeab3444bbc005dcd0cf997742cf81e468961a3f6611cbf7b8e1775474f058424a4dc1cea0ae5267c2");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463cd8f6dca118f8ee9d3f312fd383d1b9d9be3f8269da558f4932e91c20ab09870c832501ff5fb5a6c2ef86724db449b7a1b198c2ef7c4d44aced9857d16afc7ead");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648513779a7851eeeda44f8032dab43503d0cd9ef5f7503078012f1486c5bf7704d4232ef92ec3c6e079a4729b4cc202ce8d8ec080fb7a358b4c046e37b0dfe783c");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02a4ad50e5f9a1fd90faaedc1650194a0c21319b5d755a31a20d592fba4237ec39fab4b67b3a050379d4b4fb6311fa0733d373ba172ae576e321cc624844067824");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4ff4ed344fa5a7c0014b3b23b026cecc91e1a215aa34187b33b6c2c62fac21f9997ed176f34460e4c853d7ab401a3b9ba4db209fa48c65d3fe97ae7fa166cea4f4");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5240bc461232dce441bacb88c4244fe650b39e93e7b2e2e3111f9035114cf7f03912a67e4b6467152ce3704212ec0c0efb0e92d466201c5b40de4d1e2c19f820f");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc6ea56fcdad6adafeab3444bbc005dcd0cf997742cf81e468961a3f6611cbf7b8e1775474f058424a4dc1cea0ae5267c22e4ca194b25b47f6df776622b3bdbca3");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463cd8f6dca118f8ee9d3f312fd383d1b9d9be3f8269da558f4932e91c20ab09870c832501ff5fb5a6c2ef86724db449b7a1b198c2ef7c4d44aced9857d16afc7ead7aff009cd3853f68d20a94535c4a97f0");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648513779a7851eeeda44f8032dab43503d0cd9ef5f7503078012f1486c5bf7704d4232ef92ec3c6e079a4729b4cc202ce8d8ec080fb7a358b4c046e37b0dfe783c4a28b04e3793336d6287bb30984e88ed");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02a4ad50e5f9a1fd90faaedc1650194a0c21319b5d755a31a20d592fba4237ec39fab4b67b3a050379d4b4fb6311fa0733d373ba172ae576e321cc624844067824fa286be31ee4369272a3b0bbf1cbd45b");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4ff4ed344fa5a7c0014b3b23b026cecc91e1a215aa34187b33b6c2c62fac21f9997ed176f34460e4c853d7ab401a3b9ba4db209fa48c65d3fe97ae7fa166cea4f4e9db3d7353ccb0eed6f748e881dc67fb");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5240bc461232dce441bacb88c4244fe650b39e93e7b2e2e3111f9035114cf7f03912a67e4b6467152ce3704212ec0c0efb0e92d466201c5b40de4d1e2c19f820f912b74a3983d14a2b3b580b74c1de5a3");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc6ea56fcdad6adafeab3444bbc005dcd0cf997742cf81e468961a3f6611cbf7b8e1775474f058424a4dc1cea0ae5267c22e4ca194b25b47f6df776622b3bdbca36db45e35e513cb167406a6f9d26cf583");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463cd8f6dca118f8ee9d3f312fd383d1b9d9be3f8269da558f4932e91c20ab09870c832501ff5fb5a6c2ef86724db449b7a1b198c2ef7c4d44aced9857d16afc7ead7aff009cd3853f68d20a94535c4a97f06ea16b67a14e80a0573457e1f9999da0");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648513779a7851eeeda44f8032dab43503d0cd9ef5f7503078012f1486c5bf7704d4232ef92ec3c6e079a4729b4cc202ce8d8ec080fb7a358b4c046e37b0dfe783c4a28b04e3793336d6287bb30984e88ed272a64ed0ca6b0997a5868be15c5f32f");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02a4ad50e5f9a1fd90faaedc1650194a0c21319b5d755a31a20d592fba4237ec39fab4b67b3a050379d4b4fb6311fa0733d373ba172ae576e321cc624844067824fa286be31ee4369272a3b0bbf1cbd45b0997237e89b7b774500155d8ed7b6d4a");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4ff4ed344fa5a7c0014b3b23b026cecc91e1a215aa34187b33b6c2c62fac21f9997ed176f34460e4c853d7ab401a3b9ba4db209fa48c65d3fe97ae7fa166cea4f4e9db3d7353ccb0eed6f748e881dc67fb604a0e091fe115fe8fa8fa0a56fff42c");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5240bc461232dce441bacb88c4244fe650b39e93e7b2e2e3111f9035114cf7f03912a67e4b6467152ce3704212ec0c0efb0e92d466201c5b40de4d1e2c19f820f912b74a3983d14a2b3b580b74c1de5a37a4584dd77caa75d2396e1af61229a2d");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc6ea56fcdad6adafeab3444bbc005dcd0cf997742cf81e468961a3f6611cbf7b8e1775474f058424a4dc1cea0ae5267c22e4ca194b25b47f6df776622b3bdbca36db45e35e513cb167406a6f9d26cf583d44e79d50281f791c0d584efb6498add");
                yield return new(Key: "01020304", Msg: "", Custom: "", Mac: "6cabb97417fd24ea21414e4e3a1fbbfcae9071991b1120cf140f690fcedbb53a8ec1ec4c24b4d312813742c91862ea4e6232634e5404ccf98b72d87e60ce1542e314a391b10f2d3e349876f58ce3e7aa84ee355b76cecc3b57ceb0ce99b0788e9e671c5af830d8b88f4ec183b7e7ca97df75963e5591a42ec54a3b931c8422d996f7198d8706330acd537411d320fad9fe094d387373463cd8f6dca118f8ee9d3f312fd383d1b9d9be3f8269da558f4932e91c20ab09870c832501ff5fb5a6c2ef86724db449b7a1b198c2ef7c4d44aced9857d16afc7ead7aff009cd3853f68d20a94535c4a97f06ea16b67a14e80a0573457e1f9999da0f5340fdd489cac77250746aae9b16cb2");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "", Mac: "d886652e9e8b03a0dd40c14ff0a0d8a7bce3cc9da13a9015e48266c19cc9dce65e88ef0a09eadbf244668358b544176f5e19b97b208e020d9f5e210728964caa3dbb7a7c10b3b412b749034d2af56a67c8f24a2fcb15df5e3b8ad3d730a82038757ea42e469ce1bc3c6509df7893ed43b2a75f06cf843d385b5e0324ee96141e063e5dc31346e566a947230ad5af90458732698d91f10648513779a7851eeeda44f8032dab43503d0cd9ef5f7503078012f1486c5bf7704d4232ef92ec3c6e079a4729b4cc202ce8d8ec080fb7a358b4c046e37b0dfe783c4a28b04e3793336d6287bb30984e88ed272a64ed0ca6b0997a5868be15c5f32fea0cfcd6a8a581ade3db739e825eac23");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "", Mac: "ac6d2f78ab62f4539a501f27c5b34fcf57c7431d9d5dd1028e3fabb9ab15d9fef166d384332d0f4461b695258241bf250bbb96ecc42c25dbeec9b30e76d1faab822d49918b0e9021f0578a4d1ed26ed5c00a0102e9b253808dbd895882ec17cc9e80d361e6ec398224f0c78c0498d8fd22789b0c740fec933e25ab2affd0c6709092a81c9dd5a3aa151ebd91b753aea4715b73063fdacf02a4ad50e5f9a1fd90faaedc1650194a0c21319b5d755a31a20d592fba4237ec39fab4b67b3a050379d4b4fb6311fa0733d373ba172ae576e321cc624844067824fa286be31ee4369272a3b0bbf1cbd45b0997237e89b7b774500155d8ed7b6d4a2b78005d520dd2bf2eab5a2c8415aba8");
                yield return new(Key: "01020304", Msg: "", Custom: "0102030405060708", Mac: "e524220c684f9ca26ff3c774457922f53879ebe3c4b82bb750990a655dc2acf9b5159c1dbaf3c2a2c0ddb8de84c5e5c75a21d9bacdf413359ab2ec1b3f390f2ecce2721e50a7e87ddf2e38e0716cfc3212a8ff56cb491bf236102cefe97a90c3be9587f1bd686a8c0a237b55264ba6d7cf84ccd20fe51b39201f5a0609f8f95528e4bd7a1b521381b994cdbfd400ec40333ed0fcbb61df4ff4ed344fa5a7c0014b3b23b026cecc91e1a215aa34187b33b6c2c62fac21f9997ed176f34460e4c853d7ab401a3b9ba4db209fa48c65d3fe97ae7fa166cea4f4e9db3d7353ccb0eed6f748e881dc67fb604a0e091fe115fe8fa8fa0a56fff42c84164bf99ed94b78fcd98ba6d13dd494");
                yield return new(Key: "01020304", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "fe7f28d3dffeb7acdb6c6d165f75d6ac25b6a97e10803846ca00b8b93a658296386e698bc99db8708ba19d912352318c1ed8d8ac3b73a6f11b9754cd0bfbbc71e14409598028160aa704530b38784ae147a06edc1e84fa6589627649707f15dc900b7102ccf97b93316b7b6fc8c9bab155e40c4d02c32cc4ad2d5d70710e0dedef99d5d63e3924047a39838061a7f07d2842c4323c90c1c5240bc461232dce441bacb88c4244fe650b39e93e7b2e2e3111f9035114cf7f03912a67e4b6467152ce3704212ec0c0efb0e92d466201c5b40de4d1e2c19f820f912b74a3983d14a2b3b580b74c1de5a37a4584dd77caa75d2396e1af61229a2df0ca6ceef9dd47e9c2c4a5fafcebdeca");
                yield return new(Key: "01020304", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "5552b9505e8dbc3aad6991a80513b176e03f910d79f1a61d0e395759eda3c9167c4fd35dc85cc15bd36416224baa741b3c9c1d2748021fc6cda207fece3ffea10259133a91b88f42e661ea073c83484147c082a57b99502df9fa6db34063c94f9e79a919058a3a87a077e89fbac60de4d9f8018d2d6d990afed3f57683880112ed23e5df9598544806ec92ab78f124cf86c2c83b216b71cc6ea56fcdad6adafeab3444bbc005dcd0cf997742cf81e468961a3f6611cbf7b8e1775474f058424a4dc1cea0ae5267c22e4ca194b25b47f6df776622b3bdbca36db45e35e513cb167406a6f9d26cf583d44e79d50281f791c0d584efb6498add7ea2802d8a686e19fda817f5abe33c58");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "17");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed610");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f584");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c0582");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc0178669");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e691");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea54");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a498");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef513353");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a89");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba9854");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db56936");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989eb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec257");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd85");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c90215");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea095e530401f2e4b4e78cfea3cb480d169");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc49ff68dc4a6287524a90c4796139ba2f6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d645f89513671473eb393cc22df3dc91fd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677bf9c713e48a81a5dde4cfbfe9e8ae3840");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c9021505865964148a1ec0da46d684b5a49899");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9284958dd521766329b2f0a9ed5852654");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea095e530401f2e4b4e78cfea3cb480d1690f2908e6a951d07c0465c351311b59e6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc49ff68dc4a6287524a90c4796139ba2f63a058136e055bf795983789be8f55bbc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d645f89513671473eb393cc22df3dc91fd5fc59c6b4ea01d65bb37dc9959719929");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677bf9c713e48a81a5dde4cfbfe9e8ae3840c548c2f81367806439640fa75de1af93");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c9021505865964148a1ec0da46d684b5a4989975ee2d34b7cb16c163250c2efc541062");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9284958dd521766329b2f0a9ed58526545ba5db1b51dd3b6020b256326bd3f469");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea095e530401f2e4b4e78cfea3cb480d1690f2908e6a951d07c0465c351311b59e68ee9b1ba14168c028ed5e1784638ce9b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc49ff68dc4a6287524a90c4796139ba2f63a058136e055bf795983789be8f55bbcbe0b14c0b750a41e8b23e5e2a59b11fb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d645f89513671473eb393cc22df3dc91fd5fc59c6b4ea01d65bb37dc99597199294e3b447ebd1ea49ca22820f8c2e14f69");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677bf9c713e48a81a5dde4cfbfe9e8ae3840c548c2f81367806439640fa75de1af933d33fc9628e36b4ee7bfc22222558842");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c9021505865964148a1ec0da46d684b5a4989975ee2d34b7cb16c163250c2efc5410628c7ed6a13534890e7e75f17e13a6b4e7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9284958dd521766329b2f0a9ed58526545ba5db1b51dd3b6020b256326bd3f469c16a8000621a10c55c47775000d65d91");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea095e530401f2e4b4e78cfea3cb480d1690f2908e6a951d07c0465c351311b59e68ee9b1ba14168c028ed5e1784638ce9bb8f2a51167ac47e2f28fcd0ec87cf49f");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc49ff68dc4a6287524a90c4796139ba2f63a058136e055bf795983789be8f55bbcbe0b14c0b750a41e8b23e5e2a59b11fb134520b634bfb3f248e5e1372bb6a39d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d645f89513671473eb393cc22df3dc91fd5fc59c6b4ea01d65bb37dc99597199294e3b447ebd1ea49ca22820f8c2e14f69b2b57a10976aa9d1055a4c25c4b755cc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677bf9c713e48a81a5dde4cfbfe9e8ae3840c548c2f81367806439640fa75de1af933d33fc9628e36b4ee7bfc22222558842eda8b4a10adb3714a68d3c1e2e497d16");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c9021505865964148a1ec0da46d684b5a4989975ee2d34b7cb16c163250c2efc5410628c7ed6a13534890e7e75f17e13a6b4e78f28968d54881a55070c084774aab726");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9284958dd521766329b2f0a9ed58526545ba5db1b51dd3b6020b256326bd3f469c16a8000621a10c55c47775000d65d9163a98abeba1dd8ec3b2d295ae73155f3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea095e530401f2e4b4e78cfea3cb480d1690f2908e6a951d07c0465c351311b59e68ee9b1ba14168c028ed5e1784638ce9bb8f2a51167ac47e2f28fcd0ec87cf49f95a011fbc4f4162eb124875c9ac318bc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc49ff68dc4a6287524a90c4796139ba2f63a058136e055bf795983789be8f55bbcbe0b14c0b750a41e8b23e5e2a59b11fb134520b634bfb3f248e5e1372bb6a39d88f0eb7e6fc47d91feaf6ff70c08c42d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d645f89513671473eb393cc22df3dc91fd5fc59c6b4ea01d65bb37dc99597199294e3b447ebd1ea49ca22820f8c2e14f69b2b57a10976aa9d1055a4c25c4b755cc37e3aaa130620b100d47dfae3f7382c5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677bf9c713e48a81a5dde4cfbfe9e8ae3840c548c2f81367806439640fa75de1af933d33fc9628e36b4ee7bfc22222558842eda8b4a10adb3714a68d3c1e2e497d1660cc8a3bb21f8b8b17c50048cb38aba9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c9021505865964148a1ec0da46d684b5a4989975ee2d34b7cb16c163250c2efc5410628c7ed6a13534890e7e75f17e13a6b4e78f28968d54881a55070c084774aab7267e8abb91d35f6bffadf1da97c66f0047");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9284958dd521766329b2f0a9ed58526545ba5db1b51dd3b6020b256326bd3f469c16a8000621a10c55c47775000d65d9163a98abeba1dd8ec3b2d295ae73155f3e3cc60611d396f5590aee840315b1287");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea095e530401f2e4b4e78cfea3cb480d1690f2908e6a951d07c0465c351311b59e68ee9b1ba14168c028ed5e1784638ce9bb8f2a51167ac47e2f28fcd0ec87cf49f95a011fbc4f4162eb124875c9ac318bcc11437db1170dbf5113dadf459d9dfe0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc49ff68dc4a6287524a90c4796139ba2f63a058136e055bf795983789be8f55bbcbe0b14c0b750a41e8b23e5e2a59b11fb134520b634bfb3f248e5e1372bb6a39d88f0eb7e6fc47d91feaf6ff70c08c42d3ea9b614023c5a369ca4b4cd2de1d076");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d645f89513671473eb393cc22df3dc91fd5fc59c6b4ea01d65bb37dc99597199294e3b447ebd1ea49ca22820f8c2e14f69b2b57a10976aa9d1055a4c25c4b755cc37e3aaa130620b100d47dfae3f7382c5d9a6a2b766482672b47c5374c01c7549");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677bf9c713e48a81a5dde4cfbfe9e8ae3840c548c2f81367806439640fa75de1af933d33fc9628e36b4ee7bfc22222558842eda8b4a10adb3714a68d3c1e2e497d1660cc8a3bb21f8b8b17c50048cb38aba92de75751002ddb4b87c0def448c2ebe4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c9021505865964148a1ec0da46d684b5a4989975ee2d34b7cb16c163250c2efc5410628c7ed6a13534890e7e75f17e13a6b4e78f28968d54881a55070c084774aab7267e8abb91d35f6bffadf1da97c66f00478db77b0553aba51ff0d297ee3eff2525");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9284958dd521766329b2f0a9ed58526545ba5db1b51dd3b6020b256326bd3f469c16a8000621a10c55c47775000d65d9163a98abeba1dd8ec3b2d295ae73155f3e3cc60611d396f5590aee840315b12878b01bdf7174afbffb898dbdd0d34f5d1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "", Mac: "f59adf11d770ff7e21fc406879d1fafdec7bc26ad581f5840fc0ca1fae19e6c5ad502b1a7271e6911623be7bdf9c050e863c1b7a659660ce178cb6d27905f4e3ad9c91a8d01dcb4e81f02a412983aa3ddddc83504f0d471e782f5fcfca4e15f139d9d7b8213e6747f3706d7f50cd0533ad294acf2d03401c14345388e9a1aaa3a6246b52102fb87ff03261c2e880445867519d98e280eea095e530401f2e4b4e78cfea3cb480d1690f2908e6a951d07c0465c351311b59e68ee9b1ba14168c028ed5e1784638ce9bb8f2a51167ac47e2f28fcd0ec87cf49f95a011fbc4f4162eb124875c9ac318bcc11437db1170dbf5113dadf459d9dfe05db2f4b4fdbc9c4e636fcfc54f49db3a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "", Mac: "ac5d87c17e695b9feda18ddc21799807588b22f192fab8aa8b9b38ae956554700c167fb936155bf257faf6d82b36bed8ffe28d7a45c2a4987fcea63a41c18aee4d296d2ab9584b2cd75564fecc03dceacd5a832571a49c8b21730939070dabfe4be7b6bd3db569364d6fbbb23388f2a11855de9b8cdd8974c9f1381a0dafe62b1668a46274205ac09c7d77f51ce7bf5ec22f7945bc689dc49ff68dc4a6287524a90c4796139ba2f63a058136e055bf795983789be8f55bbcbe0b14c0b750a41e8b23e5e2a59b11fb134520b634bfb3f248e5e1372bb6a39d88f0eb7e6fc47d91feaf6ff70c08c42d3ea9b614023c5a369ca4b4cd2de1d076f8fccc8a5ca82c6078db5ea6177908cd");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "", Mac: "1719f4dabc8ed6109814aaf7cbf8c732ab1f8b9d66df11e49821c886576abea60d744fd7963dea548cc148ad6c56b9378e68103bef5133534ca5ef6f96bf1930f7d66eef5bba985469ca45e19e2620d45f3ab29100eb8ee0e291dbc6c5859d66c9381131c3acd82e892acddf37980d04bcffe0a2f0ed558bb2e1644a764a28b64d47e1ca567e1ea540c2fc7e315d8b24e7c38d313399c6d645f89513671473eb393cc22df3dc91fd5fc59c6b4ea01d65bb37dc99597199294e3b447ebd1ea49ca22820f8c2e14f69b2b57a10976aa9d1055a4c25c4b755cc37e3aaa130620b100d47dfae3f7382c5d9a6a2b766482672b47c5374c01c754905942bc1f6632a5fc8cebe1b4b739a29");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "", Custom: "0102030405060708", Mac: "a13edaa89c7daac4fd0ea338f987d09cd2a4adbe767c05821e0f6a6741c7d5e6cb877cdab326b00cb81e35541db97df9be4aafb7f3407a891c7b415ef20518d7c80e7c4a0fe266ee57af6aec507642331fd9b4521c13cdba53d476672081c23199a3935427cafff8bb0687772c5fd48f5720f871fa9a079c5cb25ffcba283b9dacd1f2de75cb0269b15ee64687656cdc1db7f6c75eff677bf9c713e48a81a5dde4cfbfe9e8ae3840c548c2f81367806439640fa75de1af933d33fc9628e36b4ee7bfc22222558842eda8b4a10adb3714a68d3c1e2e497d1660cc8a3bb21f8b8b17c50048cb38aba92de75751002ddb4b87c0def448c2ebe483eca50df60cdddfb18c3a08719f6dad");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "39b04b4b6089ede29484deb5c8a1548e412d93fdc017866952c8e1d7fc58f0cfa5ee094af8bb29cd602ba90a4004f2dc7b3af9b131744fa5ebd5359a1a09d5e45b27aa86ec982c1d3e5381b558ee5c77a6d6538ae8133d70d84331f9cbf5ca31a1cd65865ca1d3b6fe4608f0a46f42844629c4ce055ec2578fcb6da048512a4df9a8ed14cb4fbd853146a04d8bfba9411445f8c723c9021505865964148a1ec0da46d684b5a4989975ee2d34b7cb16c163250c2efc5410628c7ed6a13534890e7e75f17e13a6b4e78f28968d54881a55070c084774aab7267e8abb91d35f6bffadf1da97c66f00478db77b0553aba51ff0d297ee3eff252571769cec502f617efb1158b02bc20ea3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f10", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "0c0c4bd378e08424f21645be9e66be4c65fd084193aaa137ff57c5a8b774f915ab120828483a6fef2ea3fb4eeb675ee36d853072989ceb40375952543a6796f9f36a832b6d13574c238efeda6121d144cff55013604be1b6c29a4b8760b3e11fc826a2cca1c989ebb152ec4d7222d83f07e52c7761a2e6b6ce975b540c299e1729aa46de70537b7afdc8f91a7793785bd4d783344d5b6dd9284958dd521766329b2f0a9ed58526545ba5db1b51dd3b6020b256326bd3f469c16a8000621a10c55c47775000d65d9163a98abeba1dd8ec3b2d295ae73155f3e3cc60611d396f5590aee840315b12878b01bdf7174afbffb898dbdd0d34f5d1d80d87ca77eb34b5825b5611f36e95f8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "86");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad29");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e559434");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc224200");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd10228623");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc65");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd064");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a33673");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc17");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e2532434");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c47");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c0555342");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add4575");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c16");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb12");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa23");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e085");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5d");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8dcc9279e31a3b3d04d7f6ba7ffccc1a1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e5f0dc88a19fd613a392c0d3192599c15");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c52073194147a073119c4882bdba607d4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0235c859dbcdd1433fe0ecfdfd8576153");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac1718ae7c196ebe25639b450f82d879ab");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5da7ee98ad3df03b2b14bef6fefb050529");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8dcc9279e31a3b3d04d7f6ba7ffccc1a1e5f83413ee6c05c3dc22649381a6e059");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e5f0dc88a19fd613a392c0d3192599c158038bdc6100df74eff61dc3c258ec572");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c52073194147a073119c4882bdba607d49f0549c4ff6f2ee87ba8ff7dfea68607");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0235c859dbcdd1433fe0ecfdfd857615345eb1e36db72d3643132fe89576192ba");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac1718ae7c196ebe25639b450f82d879ab391eb312b8a3ed09772f1a9b1ea3e846");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5da7ee98ad3df03b2b14bef6fefb05052927c7ef80323269f7f87191134cfde211");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8dcc9279e31a3b3d04d7f6ba7ffccc1a1e5f83413ee6c05c3dc22649381a6e0597a5a6d06e893bb89569b445f8f7333eb");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e5f0dc88a19fd613a392c0d3192599c158038bdc6100df74eff61dc3c258ec5723deed760847771f31cd21052fc9029f8");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c52073194147a073119c4882bdba607d49f0549c4ff6f2ee87ba8ff7dfea686077b5c169b184d96aaa358d36b8c2ce67e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0235c859dbcdd1433fe0ecfdfd857615345eb1e36db72d3643132fe89576192ba22d26016010c64ca0c584ad150f64a08");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac1718ae7c196ebe25639b450f82d879ab391eb312b8a3ed09772f1a9b1ea3e846129093e4ec739c94e411845f71698278");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5da7ee98ad3df03b2b14bef6fefb05052927c7ef80323269f7f87191134cfde211ac360843c277428dd28bc532cfbc5151");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8dcc9279e31a3b3d04d7f6ba7ffccc1a1e5f83413ee6c05c3dc22649381a6e0597a5a6d06e893bb89569b445f8f7333ebe0619d2fce7e2ef9792431ad032beb06");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e5f0dc88a19fd613a392c0d3192599c158038bdc6100df74eff61dc3c258ec5723deed760847771f31cd21052fc9029f8df45d9a5f2cf779736c8ad4d4f8a6c4e");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c52073194147a073119c4882bdba607d49f0549c4ff6f2ee87ba8ff7dfea686077b5c169b184d96aaa358d36b8c2ce67e4dd34194ecb9cf934f680e68a436ff50");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0235c859dbcdd1433fe0ecfdfd857615345eb1e36db72d3643132fe89576192ba22d26016010c64ca0c584ad150f64a0872dea79063ab5143ac0a32f5192c1ba7");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac1718ae7c196ebe25639b450f82d879ab391eb312b8a3ed09772f1a9b1ea3e846129093e4ec739c94e411845f71698278d01ed9f98d4d5322fffdbe42d48768b3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5da7ee98ad3df03b2b14bef6fefb05052927c7ef80323269f7f87191134cfde211ac360843c277428dd28bc532cfbc5151b96df3fdb8ef4d6a6911a54923a7bd6a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8dcc9279e31a3b3d04d7f6ba7ffccc1a1e5f83413ee6c05c3dc22649381a6e0597a5a6d06e893bb89569b445f8f7333ebe0619d2fce7e2ef9792431ad032beb06449fb1dda9fc4a21cde0ac131ef3ba8b");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e5f0dc88a19fd613a392c0d3192599c158038bdc6100df74eff61dc3c258ec5723deed760847771f31cd21052fc9029f8df45d9a5f2cf779736c8ad4d4f8a6c4ef521748613903bcfe7802240e7142781");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c52073194147a073119c4882bdba607d49f0549c4ff6f2ee87ba8ff7dfea686077b5c169b184d96aaa358d36b8c2ce67e4dd34194ecb9cf934f680e68a436ff5004149d55191ee2df4480afb459e1f8d4");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0235c859dbcdd1433fe0ecfdfd857615345eb1e36db72d3643132fe89576192ba22d26016010c64ca0c584ad150f64a0872dea79063ab5143ac0a32f5192c1ba77abe1eb9fe45b865850853d43616e5ab");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac1718ae7c196ebe25639b450f82d879ab391eb312b8a3ed09772f1a9b1ea3e846129093e4ec739c94e411845f71698278d01ed9f98d4d5322fffdbe42d48768b319014adde53353b8680c7a449d409e32");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5da7ee98ad3df03b2b14bef6fefb05052927c7ef80323269f7f87191134cfde211ac360843c277428dd28bc532cfbc5151b96df3fdb8ef4d6a6911a54923a7bd6ac88f3ce58a91a9b40bf1fa6c2c3f951a");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8dcc9279e31a3b3d04d7f6ba7ffccc1a1e5f83413ee6c05c3dc22649381a6e0597a5a6d06e893bb89569b445f8f7333ebe0619d2fce7e2ef9792431ad032beb06449fb1dda9fc4a21cde0ac131ef3ba8bfe3f62942a7c8331bd9b7c482b8d0299");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e5f0dc88a19fd613a392c0d3192599c158038bdc6100df74eff61dc3c258ec5723deed760847771f31cd21052fc9029f8df45d9a5f2cf779736c8ad4d4f8a6c4ef521748613903bcfe7802240e714278128387b9de58b5c63058de74f83f53486");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c52073194147a073119c4882bdba607d49f0549c4ff6f2ee87ba8ff7dfea686077b5c169b184d96aaa358d36b8c2ce67e4dd34194ecb9cf934f680e68a436ff5004149d55191ee2df4480afb459e1f8d45fafe1794eaefd479a7964139cce4534");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0235c859dbcdd1433fe0ecfdfd857615345eb1e36db72d3643132fe89576192ba22d26016010c64ca0c584ad150f64a0872dea79063ab5143ac0a32f5192c1ba77abe1eb9fe45b865850853d43616e5ab0cd61519984b8d670282a27bb66b0198");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac1718ae7c196ebe25639b450f82d879ab391eb312b8a3ed09772f1a9b1ea3e846129093e4ec739c94e411845f71698278d01ed9f98d4d5322fffdbe42d48768b319014adde53353b8680c7a449d409e327d57c83086f2b7f7cf21e8c21920fdef");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5da7ee98ad3df03b2b14bef6fefb05052927c7ef80323269f7f87191134cfde211ac360843c277428dd28bc532cfbc5151b96df3fdb8ef4d6a6911a54923a7bd6ac88f3ce58a91a9b40bf1fa6c2c3f951af1a749f31231b30e8659f8947d058ee9");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "", Mac: "6b7a9c054701bb10107afaf03c373e7f38d8adf7abaa89f6f96ccac6a17171be327ce3183e55943489dc131158c2cb6fed25fb00ec9cddff0da5d8866d49a88ef0c7177299010236254819da21f54c082c031d8347a336732fe1615b7b1abbe832617764e253243451284b939e5dfc9bad87fda9032b079bf56435240ec18233e7495111e938ec0ba736ee22ab6f06366ea7d656c4f553a8dcc9279e31a3b3d04d7f6ba7ffccc1a1e5f83413ee6c05c3dc22649381a6e0597a5a6d06e893bb89569b445f8f7333ebe0619d2fce7e2ef9792431ad032beb06449fb1dda9fc4a21cde0ac131ef3ba8bfe3f62942a7c8331bd9b7c482b8d029911689144455d65c886cc7c7b356649f3");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "", Mac: "ef7d1908038a05c4397826856eed0f842e50bf50800e840db789a1dd409df0ea5c8181124a35e78a37310f27c8699d7109087894e923fc6561655bfc67ae5bb2bfc4b3bab882d9b4c007d092f0ad3696e4287ab47fe3bc1717ed02936ddbf7a2eabd020228b3793b0ab81f1ae7d69c29a680d179fc9934ce3015ffd57280c667fbb08c289df37c167aece0c7dad38779bb67d8e8d75e977e5f0dc88a19fd613a392c0d3192599c158038bdc6100df74eff61dc3c258ec5723deed760847771f31cd21052fc9029f8df45d9a5f2cf779736c8ad4d4f8a6c4ef521748613903bcfe7802240e714278128387b9de58b5c63058de74f83f53486839fd1e2701dd800bd44979c779ed351");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "", Mac: "43ef4be51b2d5be876dbfb1822e8048cbbcdaa80dfb5c5cc40f71350b5f34863671ac163dc2242005c88890cea673a13ff057b8c28aa20c2cd16153b6e5b399b7d9c0ad5ce940182187849fceee02c524b6e6bc51d5a2000a3cfb33d00e52f244bf64da032ec4c4784b9a00edf5074ff310b8102af8cd054d105c84b42d2d09780319ab04d18cb126381687156c6bc32a9178e8e1a7c142c52073194147a073119c4882bdba607d49f0549c4ff6f2ee87ba8ff7dfea686077b5c169b184d96aaa358d36b8c2ce67e4dd34194ecb9cf934f680e68a436ff5004149d55191ee2df4480afb459e1f8d45fafe1794eaefd479a7964139cce45345b335474333cb36914ef501367e9d8ef");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "", Custom: "0102030405060708", Mac: "b6a3708007e603c02982eac38a3c1a64e7f5035369b44244a6b4ea079dd4dcdc2fedab6ff3f81b84f165645e38d97b9834a1c5ca594ce9ab9026951010496419c4bbbc102172439cd8a1ad5c63f60b7701161a2a267f5ec9d7fff2fc5d37ba696acb8428c05553424139d8867837a8cfdec487197ec744c0d9c4c1c4583664f9685a2c14a370aa230836a4d1956c6bc02035e6f7f41eebc0235c859dbcdd1433fe0ecfdfd857615345eb1e36db72d3643132fe89576192ba22d26016010c64ca0c584ad150f64a0872dea79063ab5143ac0a32f5192c1ba77abe1eb9fe45b865850853d43616e5ab0cd61519984b8d670282a27bb66b0198a063884cc1bf5f3a578e13c05ff771b5");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "68656c6c6f", Custom: "0102030405060708", Mac: "865acc99bbccad298ddee7a6aeb3459bb402aaa45d20e6a9152211a7f33411fc0b791c79f163aed41f2734df69c6130595df30ceceee8fa0a733ba5e705ba0ab2603528e43fd674b5d139e00cb0738ef6c834a7f00e2bf1c3e58ea2b07801e4d6b9802634add45751bd51b1cb12f1aebd8c0b8e8010a4dd0d7bcfe8cec218f586dff65641bee51d976ff099f5ab13e678ca404bb0b776fac1718ae7c196ebe25639b450f82d879ab391eb312b8a3ed09772f1a9b1ea3e846129093e4ec739c94e411845f71698278d01ed9f98d4d5322fffdbe42d48768b319014adde53353b8680c7a449d409e327d57c83086f2b7f7cf21e8c21920fdef70a26036ff064baec34d3c81ec6954e1");
                yield return new(Key: "0102030405060708090a0b0c0d0e0f101112131415161718191a1b1c1d1e1f20", Msg: "676f6f64627965", Custom: "0102030405060708", Mac: "c6c1dfb50ded942d89d818769049cfaaeaacd012c59f4ab85798f530e107b9229b68aecd102286234398442328bfdc2494d53dbfe5cee304f4d601780f94bff2b1276162a9dfd06445a4fe55d5733918b27fb1517d11b476a08245010a16989b2fd43ee6f9be4240f3e881573117d384503457fb67c755ae7288da0f998465435c9ad9dc56c4e08563e76696bd78dcc82e284c9033c8dd5da7ee98ad3df03b2b14bef6fefb05052927c7ef80323269f7f87191134cfde211ac360843c277428dd28bc532cfbc5151b96df3fdb8ef4d6a6911a54923a7bd6ac88f3ce58a91a9b40bf1fa6c2c3f951af1a749f31231b30e8659f8947d058ee9b8a33d101b8e2b87b79df9703fddb027");

                // NIST test vectors
                // From https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/KMACXOF_samples.pdf
                yield return new NistKmacTestVector(Name: "Sample #4", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "00010203", CustomText: "My Tagged Application", Mac: "1755133F1534752AAD0748F2C706FB5C784512CAB835CD15676B16C0C6647FA96FAA7AF634A0BF8FF6DF39374FA00FAD9A39E322A7C92065A64EB1FB0801EB2B");
                yield return new NistKmacTestVector(Name: "Sample #5", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7", CustomText: "", Mac: "FF7B171F1E8A2B24683EED37830EE797538BA8DC563F6DA1E667391A75EDC02CA633079F81CE12A25F45615EC89972031D18337331D24CEB8F8CA8E6A19FD98B");
                yield return new NistKmacTestVector(Name: "Sample #6", Key:"404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F", Msg: "000102030405060708090A0B0C0D0E0F101112131415161718191A1B1C1D1E1F202122232425262728292A2B2C2D2E2F303132333435363738393A3B3C3D3E3F404142434445464748494A4B4C4D4E4F505152535455565758595A5B5C5D5E5F606162636465666768696A6B6C6D6E6F707172737475767778797A7B7C7D7E7F808182838485868788898A8B8C8D8E8F909192939495969798999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8A9AAABACADAEAFB0B1B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3C4C5C6C7", CustomText: "My Tagged Application", Mac: "D5BE731C954ED7732846BB59DBE3A8E30F83E77A4BFF4459F2F1C2B4ECEBB8CE67BA01C62E8AB8578D2D499BD1BB276768781190020A306A97DE281DCC30305D");
            }
        }
    }
}
